package tools

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"path"
	"sort"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/sectools/internal/storage"
	"a.yandex-team.ru/security/sectools/pkg/sectools"
)

const (
	fetchTimeout = 60 * time.Second
	syncDrift    = 15 * time.Minute
	syncPeriod   = 1 * time.Minute
)

type Syncer struct {
	ctx          context.Context
	cancelFn     context.CancelFunc
	lastSync     time.Time
	mu           sync.RWMutex
	store        *storage.Storage
	privateTools []string
	tools        map[string]*Tool
	log          log.Logger
}

type Tool struct {
	Name     string
	ImageURL string
	Private  bool
	Releases map[sectools.Channel]Release
}

type Release struct {
	Version   string
	Platforms []Platform
}

type Platform struct {
	Platform     string
	Arch         string
	BinaryName   string
	Version      string
	DownloadInfo sectools.DownloadInfo
}

func NewSyncer(logger log.Logger, store *storage.Storage, privateTools []string) (*Syncer, error) {
	ctx, cancelCtx := context.WithCancel(context.Background())

	out := &Syncer{
		store:        store,
		privateTools: privateTools,
		ctx:          ctx,
		cancelFn:     cancelCtx,
		log:          logger,
	}

	return out, out.Sync(true)
}

func (s *Syncer) Sync(stopOnError bool) error {
	ctx, cancel := context.WithTimeout(s.ctx, fetchTimeout)
	defer cancel()

	//decoder, err := zstd.NewReader(nil)
	//if err != nil {
	//	return fmt.Errorf("can't create zstd reader: %w", err)
	//}

	storedTools, err := s.store.List(ctx, "")
	if err != nil {
		return fmt.Errorf("failed to list tools: %w", err)
	}

	tools := make(map[string]*Tool, len(s.tools))
	for _, storedTool := range storedTools {
		versions, err := s.store.List(ctx, storedTool+"/")
		if err != nil {
			if stopOnError {
				return err
			}

			s.log.Warn("failed to list tool versions", log.String("tool", storedTool), log.Error(err))
			continue
		}

		tool := &Tool{
			Name:     storedTool,
			ImageURL: fmt.Sprintf("/static/img/tools/%s.png", storedTool),
			Releases: make(map[sectools.Channel]Release),
		}

		for _, privTool := range s.privateTools {
			if privTool == tool.Name {
				tool.Private = true
				break
			}
		}

		for _, version := range versions {
			channel := sectools.Channel(version)
			switch channel {
			case sectools.ChannelStable, sectools.ChannelPrestable, sectools.ChannelTesting:
			default:
				continue
			}

			release, err := s.ToolRelease(ctx, storedTool, version)
			if err != nil {
				if stopOnError {
					return err
				}

				s.log.Warn("failed to fetch tool", log.String("tool", storedTool), log.Error(err))
				continue
			}

			tool.Releases[channel] = release
		}

		if len(tool.Releases) > 0 {
			tools[tool.Name] = tool
		}
	}

	s.mu.Lock()
	s.tools = tools
	s.lastSync = time.Now()
	s.mu.Unlock()

	s.log.Info("tools synced")
	return nil
}

func (s *Syncer) Start() {
	var err error
	for {
		toNextSync := time.Until(
			time.Now().Add(syncPeriod).Truncate(syncPeriod),
		)

		select {
		case <-s.ctx.Done():
			return
		case <-time.After(toNextSync):
			err = s.Sync(false)
			if err != nil {
				s.log.Error("failed to sync tools", log.Error(err))
			}
		}
	}
}

func (s *Syncer) InSync() error {
	s.mu.RLock()
	defer s.mu.RUnlock()

	sinceSync := time.Since(s.lastSync)
	if sinceSync > syncDrift {
		return fmt.Errorf("out of sync: %s (cur drift) > %s (allowed)", sinceSync, syncDrift)
	}

	return nil
}

func (s *Syncer) Shutdown() {
	s.cancelFn()
}

func (s *Syncer) Tool(toolName string) (*Tool, bool) {
	s.mu.RLock()
	defer s.mu.RUnlock()

	tool, ok := s.tools[toolName]
	return tool, ok
}

func (s *Syncer) ToolRelease(ctx context.Context, toolName, version string) (Release, error) {
	var manifestBuf bytes.Buffer
	err := s.store.Download(ctx, path.Join(toolName, version, sectools.ManifestFilename), &manifestBuf)
	if err != nil {
		return Release{}, err
	}

	var manifest sectools.Manifest
	err = json.Unmarshal(manifestBuf.Bytes(), &manifest)
	if err != nil {
		return Release{}, err
	}

	out := Release{
		Version: manifest.Version,
	}
	for platform, arches := range manifest.Binaries {
		for arch, downloadInfo := range arches {
			out.Platforms = append(out.Platforms, Platform{
				Version:      manifest.Version,
				Platform:     platform,
				Arch:         arch,
				BinaryName:   toolBinaryName(toolName, platform),
				DownloadInfo: downloadInfo,
			})
		}
	}
	sort.Slice(out.Platforms, func(i, j int) bool {
		return out.Platforms[i].Platform < out.Platforms[j].Platform
	})
	return out, nil
}

func (s *Syncer) List() []*Tool {
	s.mu.RLock()
	defer s.mu.RUnlock()

	out := make([]*Tool, 0, len(s.tools))
	for _, tool := range s.tools {
		out = append(out, tool)
	}

	sort.Slice(out, func(i, j int) bool {
		return out[i].Name < out[j].Name
	})
	return out
}

func toolBinaryName(toolName, platform string) string {
	var out strings.Builder
	out.WriteString(toolName)

	if platform == "windows" {
		out.WriteString(".exe")
	}

	return out.String()
}
