package sectools

import (
	"context"
	"crypto/tls"
	"encoding/json"
	"fmt"
	"io"
	"time"

	"github.com/go-resty/resty/v2"
	"github.com/klauspost/compress/zstd"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/security/libs/go/hashreader"
)

const (
	DefaultUpstream  = "https://tools.sec.yandex-team.ru/"
	ManifestFilename = "manifest.json"
)

type Service struct {
	httpc *resty.Client
}

func NewService(opts ...Option) (*Service, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, fmt.Errorf("can't create ca pool: %w", err)
	}

	httpc := resty.New().
		SetTLSClientConfig(&tls.Config{RootCAs: certPool}).
		SetJSONEscapeHTML(false).
		SetBaseURL(DefaultUpstream).
		SetRetryCount(5).
		SetRetryWaitTime(1 * time.Second).
		SetRetryMaxWaitTime(100 * time.Second).
		AddRetryCondition(func(rsp *resty.Response, err error) bool {
			return err != nil
		})

	svc := &Service{httpc: httpc}
	for _, o := range opts {
		o(svc)
	}

	return svc, nil
}

func (s *Service) UploadVersion(ctx context.Context, tool Tool, toolReader io.Reader) (*UploadedTool, error) {
	encR, encW := io.Pipe()
	defer func() {
		_ = encR.Close()
	}()

	encoder, err := zstd.NewWriter(encW)
	if err != nil {
		return nil, fmt.Errorf("failed to create zstd encoder: %w", err)
	}

	go func() {
		_, _ = io.Copy(encoder, toolReader)
		_ = encoder.Close()
		_ = encW.Close()
	}()

	hashedR, err := hashreader.NewHashReader(encR)
	if err != nil {
		return nil, fmt.Errorf("failed to create hash reader: %w", err)
	}

	var remoteRsp ServiceUploadRsp
	var remoteErr ServiceError
	rsp, err := s.httpc.R().
		SetContext(ctx).
		SetBody(hashedR).
		SetError(&remoteErr).
		SetResult(&remoteRsp).
		SetHeader("Content-Type", "application/zstd").
		SetPathParams(map[string]string{
			"tool":     tool.Name,
			"version":  tool.Version,
			"platform": tool.Platform,
			"arch":     tool.Arch,
		}).
		Put("/api/v2/upload/tool/{tool}/{version}/{platform}/{arch}")

	if err != nil {
		return nil, fmt.Errorf("request failed: %w", err)
	}

	if remoteErr.Msg != "" {
		return nil, &remoteErr
	}

	if !rsp.IsSuccess() {
		return nil, fmt.Errorf("request failed: non-200 status code: %s", rsp.Status())
	}

	return &UploadedTool{
		Tool: tool,
		DownloadInfo: DownloadInfo{
			URL:     remoteRsp.URL,
			FastURL: remoteRsp.FastURL,
			DumbURL: remoteRsp.DumbURL,
			Size:    hashedR.Size(),
			Hash:    hashedR.Hash(),
		},
	}, nil
}

func (s *Service) UploadManifest(ctx context.Context, toolName string, manifest Manifest) (*UploadedManifest, error) {
	encR, encW := io.Pipe()
	defer func() {
		_ = encR.Close()
	}()

	encoder := json.NewEncoder(encW)
	go func() {
		_ = encoder.Encode(manifest)
		_ = encW.Close()
	}()

	hashedR, err := hashreader.NewHashReader(encR)
	if err != nil {
		return nil, fmt.Errorf("failed to create hash reader: %w", err)
	}

	var remoteRsp ServiceUploadRsp
	var remoteErr ServiceError
	rsp, err := s.httpc.R().
		SetContext(ctx).
		SetBody(hashedR).
		SetError(&remoteErr).
		SetResult(&remoteRsp).
		SetHeader("Content-Type", "application/json").
		SetPathParams(map[string]string{
			"tool":    toolName,
			"version": manifest.Version,
		}).
		Put("/api/v2/upload/manifest/{tool}/{version}/" + ManifestFilename)

	if err != nil {
		return nil, fmt.Errorf("request failed: %w", err)
	}

	if remoteErr.Msg != "" {
		return nil, &remoteErr
	}

	if !rsp.IsSuccess() {
		return nil, fmt.Errorf("request failed: non-200 status code: %s", rsp.Status())
	}

	return &UploadedManifest{
		Manifest: manifest,
		DownloadInfo: DownloadInfo{
			URL:     remoteRsp.URL,
			FastURL: remoteRsp.FastURL,
			DumbURL: remoteRsp.DumbURL,
			Size:    hashedR.Size(),
			Hash:    hashedR.Hash(),
		},
	}, nil
}

func (s *Service) Release(ctx context.Context, toolName, version string, channel Channel) error {
	var req = struct {
		FromVersion string  `json:"from_version"`
		ToChannel   Channel `json:"to_channel"`
	}{
		FromVersion: version,
		ToChannel:   channel,
	}

	var remoteErr ServiceError
	rsp, err := s.httpc.R().
		SetContext(ctx).
		SetBody(req).
		SetError(&remoteErr).
		SetHeader("Content-Type", "application/json").
		SetPathParams(map[string]string{
			"tool": toolName,
		}).
		Post("/api/v2/release/{tool}")

	if err != nil {
		return fmt.Errorf("request failed: %w", err)
	}

	if remoteErr.Msg != "" {
		return &remoteErr
	}

	if !rsp.IsSuccess() {
		return fmt.Errorf("request failed: non-200 status code: %s", rsp.Status())
	}

	return nil
}
