package templates

import (
	"fmt"
	"io/fs"
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"
)

type Builder struct {
	toolName    string
	packageName string
	version     string
	authorName  string
	authorEmail string
	src         fs.FS
}

func NewBuilder(src fs.FS, opts ...Option) *Builder {
	out := &Builder{
		src: src,
	}

	for _, opt := range opts {
		opt(out)
	}

	return out
}

func (b *Builder) Build(dst string) error {
	var author string
	if b.authorName != "" {
		author = fmt.Sprintf("%s <%s>", b.authorName, b.authorEmail)
	}

	replacer := strings.NewReplacer(
		"@@init@@", "__init__",
		"@@tool@@", b.toolName,
		"@@pkg@@", b.packageName,
		"@@version@@", b.version,
		"@@author@@", author,
		"@@author_name@@", b.authorName,
		"@@author_email@@", b.authorEmail,
	)

	return fs.WalkDir(b.src, ".", func(path string, d fs.DirEntry, err error) error {
		if err != nil {
			return err
		}

		if path == "." {
			return nil
		}

		destPath := filepath.Join(dst, replacer.Replace(path))
		if d.IsDir() {
			return os.MkdirAll(destPath, 0o755)
		}

		data, err := fs.ReadFile(b.src, path)
		if err != nil {
			return fmt.Errorf("failed to read file %q: %w", path, err)
		}

		destData := []byte(replacer.Replace(string(data)))
		err = ioutil.WriteFile(destPath, destData, 0o644)
		if err != nil {
			return fmt.Errorf("failed to write file %q: %w", destPath, err)
		}

		return nil
	})
}
