package cli

import (
	"context"
	"encoding/json"
	"fmt"
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/sic/internal/idm"
)

var (
	listFormat      = "console"
	listSystemState = idm.SystemStateUnspecified.String()
)

var listCmd = &cobra.Command{
	Use: "list",
	PreRunE: func(cmd *cobra.Command, args []string) error {
		switch listFormat {
		case "console", "json", "jsonlines":
		default:
			return xerrors.Errorf("unsupported output format: %s", listFormat)
		}

		return nil
	},
	RunE: func(cmd *cobra.Command, args []string) error {
		idmClient, err := idm.NewClient(
			oauthToken,
			idm.WithLogger(logger),
			idm.WithDebug(verbose),
		)
		if err != nil {
			return xerrors.Errorf("failed to create IDM client: %w", err)
		}

		systems, err := idmClient.Systems(context.Background(), &idm.SystemsRequest{
			State:       idm.SystemState(listSystemState),
			PluginTypes: []idm.PluginType{idm.PluginTypeGeneric, idm.PluginTypeGenericLegacy},
		})

		switch {
		case err != nil && len(systems) == 0:
			return err
		case err != nil:
			logger.Error("failed to list IDM systems", log.Error(err))
		}

		return listOutput(systems, listFormat)
	},
}

func init() {
	flags := listCmd.PersistentFlags()
	flags.StringVar(&listFormat, "format", listFormat, "output format (console, json, jsonlines)")
	flags.StringVar(&listSystemState, "system-state", listSystemState, "target systems state (active, inactive, broken)")
	RootCmd.AddCommand(listCmd)
}

func listOutput(systems []idm.System, format string) error {
	switch format {
	case "console":
		for _, system := range systems {
			fmt.Printf("%s (https://idm.yandex-team.ru/system/%s/)\n", system.Name.String(), system.Slug)
		}
	case "json":
		out, err := json.MarshalIndent(systems, "", "  ")
		if err != nil {
			return err
		}
		_, _ = os.Stdout.Write(out)
	case "jsonlines":
		enc := json.NewEncoder(os.Stdout)
		for _, system := range systems {
			if err := enc.Encode(system); err != nil {
				return err
			}
		}
	}
	return nil
}
