package cli

import (
	"context"
	"os"

	"github.com/spf13/cobra"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/library/go/core/log"
	xzap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/oauth"
	"a.yandex-team.ru/security/sic/internal/consts"
)

var (
	logger     log.Structured
	verbose    bool
	oauthToken string
)

var RootCmd = &cobra.Command{
	Use:          "sic-cli",
	SilenceUsage: true,
	Short:        "SIC toolkit",
	PersistentPreRunE: func(cmd *cobra.Command, args []string) error {
		logLvl := log.InfoLevel
		if verbose {
			logLvl = log.DebugLevel
		}

		l, err := newLogger(logLvl)
		if err != nil {
			return xerrors.Errorf("failed to create logger: %w", err)
		}
		logger = l

		if cmd == versionCmd {
			// OK
			return nil
		}

		oauthToken = os.Getenv("SIC_TOKEN")
		if oauthToken == "" {
			// Try to get OAuth token from ssh
			oauthToken, err = oauth.GetTokenBySSH(
				context.Background(),
				consts.OAuthClientID, consts.OAuthClientSecret,
				oauth.WithLogger(logger),
			)
			if err != nil {
				logger.Error("failed to get OAuth token by SSH key", log.Error(err))
			}

			if err != nil || oauthToken == "" {
				return xerrors.Errorf("can't get oauth token from env[SIC_TOKEN] or exchange SSH key.")
			}
		}
		return nil
	},
}

func init() {
	flags := RootCmd.PersistentFlags()
	flags.BoolVar(&verbose, "verbose", verbose, "verbose mode")
}

func Execute() {
	if err := RootCmd.Execute(); err != nil {
		os.Exit(1)
	}
}

func newLogger(level log.Level) (*xzap.Logger, error) {
	return xzap.New(zap.Config{
		Level:            zap.NewAtomicLevelAt(xzap.ZapifyLevel(level)),
		Encoding:         "console",
		OutputPaths:      []string{"stderr"},
		ErrorOutputPaths: []string{"stderr"},
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:     "msg",
			LevelKey:       "level",
			TimeKey:        "ts",
			CallerKey:      "caller",
			NameKey:        "name",
			EncodeLevel:    zapcore.CapitalLevelEncoder,
			EncodeTime:     zapcore.ISO8601TimeEncoder,
			EncodeDuration: zapcore.StringDurationEncoder,
			EncodeCaller:   zapcore.ShortCallerEncoder,
		},
	})
}
