package idm

import "time"

type AuthFactor string

const (
	AuthFactorUnspecified AuthFactor = ""
	AuthFactorCert        AuthFactor = "cert"
	AuthFactorTVM         AuthFactor = "tvm"
)

func (s AuthFactor) String() string {
	return string(s)
}

type GroupPolicy string

const (
	GroupPolicyUnspecified                     GroupPolicy = ""
	GroupPolicyAware                           GroupPolicy = "aware"
	GroupPolicyUnaware                         GroupPolicy = "unaware"
	GroupPolicyUnavailable                     GroupPolicy = "unavailable"
	GroupPolicyAwareOfMembershipsWithoutLogins GroupPolicy = "aware_of_memberships_without_logins"
)

func (s GroupPolicy) String() string {
	return string(s)
}

type ServiceState string

const (
	ServiceStateUnspecified ServiceState = ""
	ServiceStateClosed      ServiceState = "closed"
	ServiceStateDeleted     ServiceState = "deleted"
	ServiceStateDevelop     ServiceState = "develop"
	ServiceStateSupported   ServiceState = "supported"
)

func (s ServiceState) String() string {
	return string(s)
}

type SystemState string

const (
	SystemStateUnspecified SystemState = ""
	SystemStateActive      SystemState = "active"
	SystemStateInactive    SystemState = "inactive"
	SystemStateBroken      SystemState = "broken"
)

func (s SystemState) String() string {
	return string(s)
}

type PluginType string

const (
	PluginTypeUnspecified   PluginType = ""
	PluginTypeGeneric       PluginType = "generic"
	PluginTypeGenericLegacy PluginType = "generic_legacy"
	PluginTypeGenericDumb   PluginType = "dumb"
)

func (p PluginType) String() string {
	return string(p)
}

type Title struct {
	En string `json:"en"`
	Ru string `json:"ru"`
}

func (t Title) String() string {
	if t.Ru != "" {
		return t.Ru
	}
	return t.En
}

type Responsible struct {
	FullName string `json:"full_name"`
	Username string `json:"username"`
}

type Service struct {
	ID              int          `json:"id"`
	CreatedAt       time.Time    `json:"created_at"`
	DescriptionHTML string       `json:"description_html"`
	DescriptionWiki string       `json:"description_wiki"`
	IsVteam         bool         `json:"is_vteam"`
	Name            string       `json:"name"`
	Parent          *Service     `json:"parent"`
	ResourceURI     string       `json:"resource_uri"`
	Root            *Service     `json:"root"`
	Slug            string       `json:"slug"`
	State           ServiceState `json:"state"`
	UpdatedAt       time.Time    `json:"updated_at"`
}

type System struct {
	BaseURL      string        `json:"base_url"`
	Name         Title         `json:"name"`
	Description  Title         `json:"description"`
	PluginType   PluginType    `json:"plugin_type"`
	Emails       []string      `json:"emails"`
	GroupPolicy  GroupPolicy   `json:"group_policy"`
	IsActive     bool          `json:"is_active"`
	IsBroken     bool          `json:"is_broken"`
	IsFavorite   bool          `json:"is_favorite"`
	IsSox        bool          `json:"is_sox"`
	Responsibles []Responsible `json:"responsibles"`
	Service      Service       `json:"service"`
	Slug         string        `json:"slug"`
	State        SystemState   `json:"state"`
	Team         []Responsible `json:"team"`
	AuthFactor   AuthFactor    `json:"auth_factor"`
	TVMid        string        `json:"tvm_id"`
}

func (s *System) HasAnyPlugin(plugins []PluginType) bool {
	if len(plugins) == 0 {
		return true
	}

	for _, plugin := range plugins {
		if s.PluginType == plugin {
			return true
		}
	}

	return false
}
