package idm

import (
	"context"
	"encoding/json"
	"net/url"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
)

type (
	idmMeta struct {
		Limit      int    `json:"limit"`
		Next       string `json:"next"`
		TotalCount int    `json:"total_count"`
	}

	SystemsRequest struct {
		// признак выключенности системы
		Broken bool
		// sox системы
		Sox bool
		// состояние системы
		State SystemState
		// нужные сервисы
		Services []string
		// нужные типы плагинов
		PluginTypes []PluginType
	}

	idmSystemsResponse struct {
		Meta    idmMeta  `json:"meta"`
		Systems []System `json:"objects"`
	}
)

func (c *Client) Systems(ctx context.Context, req *SystemsRequest) ([]System, error) {
	params := url.Values{
		"limit": {"200"},
	}

	if req.Broken {
		params.Set("is_broken", "true")
	}

	if req.Sox {
		params.Set("is_sox", "true")
	}

	if len(req.Services) > 0 {
		params.Set("service", strings.Join(req.Services, ","))
	}

	if req.State != SystemStateUnspecified {
		params.Set("state", req.State.String())
	}

	endpoint := "/api/v1/systems/?" + params.Encode()
	var systems []System

	for endpoint != "" {
		c.logger.Debug("system request", log.String("endpoint", endpoint))

		resp, err := c.httpClient.R().
			SetContext(ctx).
			Get(endpoint)
		if err != nil {
			return systems, xerrors.Errorf("failed to call IDM: %w", err)
		}

		data := new(idmSystemsResponse)
		if err = json.Unmarshal(resp.Body(), data); err != nil {
			return systems, xerrors.Errorf("failed to parse IDM response: %w", err)
		}

		endpoint = data.Meta.Next
		for _, system := range data.Systems {
			if !system.HasAnyPlugin(req.PluginTypes) {
				continue
			}

			systems = append(systems, system)
		}
	}

	return systems, nil
}
