package app

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/security/libs/go/xlock"
	"a.yandex-team.ru/security/skotty/datalens-exporter/internal/config"
	"a.yandex-team.ru/security/skotty/datalens-exporter/internal/exportdb"
	"a.yandex-team.ru/security/skotty/datalens-exporter/internal/models"
	"a.yandex-team.ru/security/skotty/datalens-exporter/internal/skottydb"
	"a.yandex-team.ru/security/skotty/datalens-exporter/internal/splunk"
	"a.yandex-team.ru/security/skotty/datalens-exporter/internal/staff"
	"a.yandex-team.ru/yt/go/yterrors"
)

const (
	batchLimit = 2000
)

type App struct {
	tickInternal       time.Duration
	staffc             *staff.Client
	exportDB           *exportdb.DB
	skottyDB           *skottydb.DB
	splunkSender       splunk.Sender
	observableDeps     map[string]struct{}
	observableUsers    map[string]struct{}
	observableServices []string
	ycloudFingerprints map[string]struct{}
	locker             xlock.Locker
	log                log.Logger
	ctx                context.Context
	cancelCtx          context.CancelFunc
	closed             chan struct{}
}

func NewApp(cfg *config.Config) (*App, error) {
	logLvl := log.InfoLevel
	if cfg.Dev {
		logLvl = log.DebugLevel
	}

	logger, err := zap.NewDeployLogger(logLvl)
	if err != nil {
		return nil, fmt.Errorf("create logger: %w", err)
	}

	locker, err := xlock.NewYTLocker(cfg.YT.Proxy, cfg.YT.Path, cfg.YT.Token)
	if err != nil {
		return nil, fmt.Errorf("create YT locker: %w", err)
	}

	staffc, err := staff.NewClient(staff.WithAuthToken(cfg.Staff.Token))
	if err != nil {
		return nil, fmt.Errorf("create staff client: %w", err)
	}

	tmvc, err := tvmtool.NewAnyClient()
	if err != nil {
		return nil, fmt.Errorf("create tvmtool client: %w", err)
	}

	exportDB, err := exportdb.NewDB(context.TODO(), tmvc, cfg.ExportDB)
	if err != nil {
		return nil, fmt.Errorf("create export db: %w", err)
	}

	skottyDB, err := skottydb.NewDB(context.TODO(), tmvc, cfg.SkottyDB)
	if err != nil {
		return nil, fmt.Errorf("create skotty db: %w", err)
	}

	var splunkSender splunk.Sender
	if cfg.Soc.Enable {
		splunkSender = splunk.NewHecSender(splunk.WithLogger(logger), splunk.WithAuthToken(cfg.Soc.HecToken))
	} else {
		splunkSender = &splunk.NopSender{}
	}

	observableDeps := make(map[string]struct{})
	for _, dep := range cfg.Soc.ObservableDepartments {
		observableDeps[dep] = struct{}{}
	}

	ycloudFingerprints := make(map[string]struct{})
	for _, fp := range cfg.YCloudKeys {
		ycloudFingerprints[fp] = struct{}{}
	}

	ctx, cancel := context.WithCancel(context.Background())
	return &App{
		tickInternal:       cfg.SyncPeriod,
		staffc:             staffc,
		exportDB:           exportDB,
		skottyDB:           skottyDB,
		splunkSender:       splunkSender,
		observableDeps:     observableDeps,
		observableServices: cfg.Soc.ObservableServices,
		observableUsers:    make(map[string]struct{}),
		ycloudFingerprints: ycloudFingerprints,
		locker:             locker,
		log:                logger,
		ctx:                ctx,
		cancelCtx:          cancel,
		closed:             make(chan struct{}),
	}, nil
}

func (a *App) Start() error {
	defer close(a.closed)

	for {
		toNextWork := time.Until(
			time.Now().Add(a.tickInternal).Truncate(a.tickInternal),
		)
		t := time.NewTimer(toNextWork)

		select {
		case <-a.ctx.Done():
			t.Stop()
		case <-t.C:
			if err := a.Export(); err != nil {
				a.log.Error("process failed", log.Error(err))
			}
		}
	}
}

func (a *App) Export() error {
	tx, err := a.locker.Lock(a.ctx)
	if err != nil {
		if yterrors.ContainsErrorCode(err, yterrors.CodeConcurrentTransactionLockConflict) {
			a.log.Info("conflict YT lock", log.String("error", err.Error()))
			return nil
		}

		return fmt.Errorf("lock acquire: %w", err)
	}
	defer func() { _ = tx.Unlock() }()

	startedAt := time.Now()
	a.log.Info("start export")
	defer func() { a.log.Info("exported", log.Duration("duration", time.Since(startedAt))) }()

	if err := a.updateObservableServices(a.ctx); err != nil {
		return fmt.Errorf("unable to update observable services: %w", err)
	}

	tokens, err := a.skottyDB.LookupYubikeyTokens(a.ctx)
	if err != nil {
		return fmt.Errorf("unable to list tokens: %w", err)
	}

	enrolledUsers := make(map[string]time.Time)
	for _, token := range tokens {
		if created, ok := enrolledUsers[token.User]; ok && created.Before(token.CreatedAt) {
			continue
		}

		enrolledUsers[token.User] = token.CreatedAt
	}

	if err := a.exportDB.DropTemporaryTables(a.ctx); err != nil {
		return fmt.Errorf("unable to drop temporary tables: %w", err)
	}

	if err := a.exportDB.CreateTemporaryTables(a.ctx); err != nil {
		return fmt.Errorf("unable to create temporary tables: %w", err)
	}

	if err := a.exportDB.InsertTotalUsers(a.ctx, len(enrolledUsers)); err != nil {
		return fmt.Errorf("unable to save total users: %w", err)
	}

	var acls []models.ACL
	var users []models.UserInfo
	var deps []models.Department
	splunkState := splunk.UsersState{
		SyncTS: startedAt.Unix(),
		Users:  make([]models.UserInfo, 0, batchLimit),
	}

	walkUsers := func() error {
		return a.staffc.WalkUsers(a.ctx, func(staffUser staff.User) error {
			enrolledAt, enrolled := enrolledUsers[staffUser.Login]
			user := models.UserInfo{
				Login:          staffUser.Login,
				HaveYCloudKeys: a.haveYCloudKeys(staffUser.SSHFingerprints),
				HaveSSHKeys:    len(staffUser.SSHFingerprints) > 0,
				Department:     staffUser.MainDepartment,
				Enrolled:       enrolled,
				EnrolledAt:     enrolledAt,
			}

			users = append(users, user)

			for _, head := range append(staffUser.Heads, staffUser.UID) {
				acls = append(acls, models.ACL{
					Login:   staffUser.Login,
					HeadUID: head,
				})
			}

			for _, dep := range staffUser.Departments {
				deps = append(deps, models.Department{
					Login:      staffUser.Login,
					Department: dep,
				})
			}

			if a.isObservableUser(staffUser) {
				splunkState.Users = append(splunkState.Users, user)
			}

			if len(users) < batchLimit && len(acls) < batchLimit && len(deps) < batchLimit {
				return nil
			}

			if err := a.exportDB.InsertUserInfos(a.ctx, users); err != nil {
				return fmt.Errorf("unable to insert users: %w", err)
			}

			if err := a.exportDB.InsertACLs(a.ctx, acls); err != nil {
				return fmt.Errorf("unable to insert acls: %w", err)
			}

			if err := a.exportDB.InsertDepartments(a.ctx, deps); err != nil {
				return fmt.Errorf("unable to insert deps: %w", err)
			}

			if err := a.splunkSender.SendUserStates(a.ctx, splunkState); err != nil {
				// ignore HEC errors due to flaky nature
				a.log.Error("unable to sync users to splunk", log.Error(err))
			}

			users = users[:0]
			acls = acls[:0]
			deps = deps[:0]
			splunkState.Users = splunkState.Users[:0]
			return nil
		})
	}

	if err := walkUsers(); err != nil {
		return err
	}

	return a.exportDB.FinalizeTables(a.ctx)
}

func (a *App) isObservableUser(user staff.User) bool {
	if _, ok := a.observableUsers[user.Login]; ok {
		return true
	}

	if _, ok := a.observableDeps[user.MainDepartmentID]; ok {
		return true
	}

	for _, dep := range user.DepartmentsIDs {
		if _, ok := a.observableDeps[dep]; ok {
			return true
		}
	}

	return false
}

func (a *App) haveYCloudKeys(keys []string) bool {
	for _, fp := range keys {
		if _, ok := a.ycloudFingerprints[fp]; ok {
			return true
		}
	}

	return false
}

func (a *App) updateObservableServices(ctx context.Context) error {
	a.observableUsers = make(map[string]struct{})
	return a.staffc.WalkABCMembers(ctx, func(user string) error {
		a.observableUsers[user] = struct{}{}
		return nil
	}, a.observableServices...)
}

func (a *App) Shutdown(ctx context.Context) error {
	a.cancelCtx()

	select {
	case <-ctx.Done():
		return ctx.Err()
	case <-a.closed:
		return nil
	}
}
