package config

import (
	"context"
	"os"
	"time"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"
)

const ShutdownDeadline = 30 * time.Minute

type YDB struct {
	Endpoint string `yaml:"endpoint"`
	Database string `yaml:"database"`
	Path     string `yaml:"path"`
}

type YT struct {
	Proxy string `yaml:"proxy"`
	Path  string `yaml:"path"`
	Token string `yaml:"token"`
}

type Staff struct {
	Token string `yaml:"token"`
}

type Roles struct {
	Admin   string `yaml:"admin"`
	RoboSSH string `yaml:"robossh"`
}

type Soc struct {
	Enable                bool     `yaml:"enable"`
	HecToken              string   `yaml:"hec_token"`
	ObservableDepartments []string `yaml:"observable_departments"`
	ObservableServices    []string `yaml:"observable_services"`
}

type Config struct {
	Dev        bool          `yaml:"dev"`
	SyncPeriod time.Duration `yaml:"sync_period"`
	SkottyDB   YDB           `yaml:"skotty_db"`
	ExportDB   YDB           `yaml:"export_db"`
	YT         YT            `yaml:"yt"`
	Staff      Staff         `yaml:"staff"`
	Soc        Soc           `yaml:"soc"`
	YCloudKeys []string      `yaml:"ycloud_keys"`
}

func Load(cfgPath string) (*Config, error) {
	cfg := &Config{
		YT: YT{
			Proxy: "locke",
			Path:  "//home/security/skotty/dev/exporter",
			Token: os.Getenv("YT_TOKEN"),
		},
		Staff: Staff{
			Token: os.Getenv("STAFF_TOKEN"),
		},
		Soc: Soc{
			Enable:   true,
			HecToken: os.Getenv("HEC_TOKEN"),
		},
		SyncPeriod: 2 * time.Hour,
		YCloudKeys: []string{
			"SHA256:xQdCsb6SvRtPo9XvY5DYNhYGjq+J3PBE5HGRcwUZKv0",
			"SHA256:WYP5pgdP85TxszLwgglTCyfBB9u4ZipK49WTJdfoxkc",
		},
	}

	loader := confita.NewLoader(env.NewBackend(), file.NewBackend(cfgPath))
	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, err
	}

	return cfg, nil
}
