package skottydb

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/libs/go/ydbtvm"
	"a.yandex-team.ru/security/skotty/datalens-exporter/internal/config"
	"a.yandex-team.ru/security/skotty/datalens-exporter/internal/models"
	"a.yandex-team.ru/security/skotty/libs/skotty"
)

const (
	ListLimit        = 999
	tokenStateActive = 1
)

type DB struct {
	sp   *table.SessionPool
	path string
}

func NewDB(ctx context.Context, tvmc tvm.Client, cfg config.YDB) (*DB, error) {
	driverConfig := &ydb.DriverConfig{
		Database: cfg.Database,
		Credentials: &ydbtvm.TvmCredentials{
			DstID:     ydbtvm.YDBClientID,
			TvmClient: tvmc,
		},
	}

	driver, err := (&ydb.Dialer{
		DriverConfig: driverConfig,
	}).Dial(ctx, cfg.Endpoint)

	if err != nil {
		return nil, fmt.Errorf("dial error: %v", err)
	}

	tableClient := table.Client{
		Driver: driver,
	}

	sp := table.SessionPool{
		IdleThreshold: 10 * time.Second,
		Builder:       &tableClient,
	}

	return &DB{
		sp:   &sp,
		path: cfg.Path,
	}, nil
}

func (d *DB) Close(ctx context.Context) error {
	return d.sp.Close(ctx)
}

func (d *DB) LookupYubikeyTokens(ctx context.Context) ([]models.YubiToken, error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	query := lookupYubikeyTokensQuery(d.path)
	var result []models.YubiToken

	requestTokens := func(ctx context.Context, yubiInfo *yubiPagination) (bool, error) {
		var res *table.Result
		err := table.Retry(ctx, d.sp,
			table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
				stmt, err := s.Prepare(ctx, query)
				if err != nil {
					return err
				}

				_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
					table.ValueParam("$tokenType", ydb.Uint8Value(uint8(skotty.TokenTypeYubikey))),
					table.ValueParam("$tokenState", ydb.Uint8Value(tokenStateActive)),
					table.ValueParam("$lastUser", ydb.UTF8Value(yubiInfo.User)),
					table.ValueParam("$lastId", ydb.UTF8Value(yubiInfo.TokenID)),
					table.ValueParam("$lastEnrollId", ydb.UTF8Value(yubiInfo.EnrollID)),
					table.ValueParam("$limit", ydb.Uint64Value(ListLimit)),
				))
				return err
			}),
		)

		if err != nil {
			return false, err
		}

		count := 0
		for res.NextSet() {
			for res.NextRow() {
				var token models.YubiToken

				// created_at, user, id, enroll_id
				res.SeekItem("created_at")
				token.CreatedAt = time.Unix(res.OInt64(), 0)

				res.NextItem()
				token.User = res.OUTF8()
				result = append(result, token)
				count++

				yubiInfo.User = token.User

				res.NextItem()
				yubiInfo.TokenID = res.OUTF8()

				res.NextItem()
				yubiInfo.EnrollID = res.OUTF8()

			}
		}

		return count >= ListLimit, res.Err()
	}

	yubiInfo := yubiPagination{}
	for {
		haveMore, err := requestTokens(ctx, &yubiInfo)
		if err != nil {
			return nil, err
		}

		if !haveMore {
			break
		}
	}

	return result, nil
}
