package splunk

import (
	"bytes"
	"context"
	"crypto/tls"
	"encoding/json"
	"fmt"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

const (
	hecEndpoint = "https://hatch.yandex.net/services/collector"
	hecRetries  = 3
)

var _ Sender = (*HecSender)(nil)

type HecSender struct {
	httpc *resty.Client
	l     log.Logger
	buf   bytes.Buffer
}

func NewHecSender(opts ...Option) *HecSender {
	s := &HecSender{
		l: &nop.Logger{},
		httpc: resty.New().
			SetRetryCount(hecRetries).
			SetDebug(true).
			SetHeader("Content-Type", "application/json"),
	}

	for _, opt := range opts {
		opt(s)
	}

	certPool, err := certifi.NewCertPool()
	if err != nil {
		s.l.Error("splunk: failed to configure TLS cert pool", log.Error(err))
	} else {
		s.httpc.SetTLSClientConfig(&tls.Config{RootCAs: certPool})
	}

	return s
}

func (s *HecSender) SendUserStates(ctx context.Context, state UsersState) error {
	if len(state.Users) == 0 {
		return nil
	}

	data, err := s.stateToEvents(state)
	if err != nil {
		return fmt.Errorf("unable to convert state: %w", err)
	}

	rsp, err := s.httpc.R().
		SetContext(ctx).
		SetBody(data).
		Post(hecEndpoint)

	if err != nil {
		return fmt.Errorf("send failed: %w", err)
	}

	if !rsp.IsSuccess() {
		return fmt.Errorf("non 200 status code: %s", rsp.Status())
	}

	return nil
}

func (s *HecSender) stateToEvents(state UsersState) ([]byte, error) {
	encoder := json.NewEncoder(&s.buf)
	defer s.buf.Reset()

	var err error
	baseEvent := BaseDataEvent{
		SyncTS: state.SyncTS,
	}

	nowTS := time.Now().Unix()
	for _, user := range state.Users {
		s.l.Info("send user to HEC", log.Any("user", user))
		err = encoder.Encode(Event{
			Timestamp: nowTS,
			Data: UserState{
				BaseDataEvent: baseEvent,
				UserState:     user,
			},
		})
		if err != nil {
			return nil, fmt.Errorf("unable to encode %q user info: %w", user.Login, err)
		}
	}

	return s.buf.Bytes(), nil
}
