package staff

import (
	"context"
	"crypto/tls"
	"fmt"
	"net/http"
	"net/url"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
)

const (
	DefaultUpstream = "https://staff-api.yandex-team.ru"
)

var ignoredGroups = map[string]struct{}{
	"yandex": {},
}

type Client struct {
	httpc    *resty.Client
	upstream string
}

func NewClient(opts ...Option) (*Client, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, fmt.Errorf("can't create ca pool: %w", err)
	}

	httpc := resty.New().
		SetTLSClientConfig(&tls.Config{RootCAs: certPool}).
		SetJSONEscapeHTML(false).
		SetHeader("Content-Type", "application/json").
		SetRetryCount(3).
		SetRetryWaitTime(100 * time.Millisecond).
		SetRetryMaxWaitTime(3 * time.Second).
		AddRetryCondition(func(rsp *resty.Response, err error) bool {
			return err != nil || rsp.StatusCode() == http.StatusInternalServerError
		})

	c := &Client{
		httpc:    httpc,
		upstream: DefaultUpstream,
	}
	for _, o := range opts {
		o(c)
	}

	return c, nil
}

func (c *Client) WalkUsers(ctx context.Context, fn func(user User) error) error {
	query := url.Values{
		"_limit":                {"500"},
		"_fields":               {"login,uid,chiefs.uid,department_group.url,department_group.name,department_group.ancestors,keys.fingerprint_sha256"},
		"official.is_dismissed": {"false"},
		"official.affiliation":  {"yandex"},
		"official.is_robot":     {"false"},
	}
	uri := fmt.Sprintf("%s/v3/persons?%s", strings.TrimRight(c.upstream, "/"), query.Encode())

	for uri != "" {
		var errResult staffErrorResponse
		var result staffPersonsResponse
		rsp, err := c.httpc.R().
			SetContext(ctx).
			SetResult(&result).
			SetError(&errResult).
			ForceContentType("application/json").
			Get(uri)

		if err != nil {
			return fmt.Errorf("request failed while request %q: %w", uri, err)
		}

		if errResult.Msg != "" {
			return fmt.Errorf("remote error on request %q: %s", uri, errResult.Msg)
		}

		if !rsp.IsSuccess() {
			return fmt.Errorf("request failed: non-200 status code for request %q: %s", uri, rsp.Status())
		}

		for _, u := range result.Result {
			heads := make([]uint64, len(u.Chiefs))
			for i, c := range u.Chiefs {
				heads[i] = c.UID
			}

			deps := make([]string, 0, len(u.DepartmentGroup.Ancestors)+1)
			depsIDs := make([]string, 0, len(u.DepartmentGroup.Ancestors)+1)
			for _, dep := range append(u.DepartmentGroup.Ancestors, u.DepartmentGroup.staffDepartment) {
				if _, ok := ignoredGroups[dep.URL]; ok {
					continue
				}

				deps = append(deps, fmt.Sprintf("%s (%s)", dep.Name, dep.URL))
				depsIDs = append(depsIDs, dep.URL)
			}

			staffKeys := make([]string, len(u.Keys))
			for i, key := range u.Keys {
				staffKeys[i] = key.Fingerprint
			}

			err = fn(User{
				Login:            u.Login,
				UID:              u.UID,
				SSHFingerprints:  staffKeys,
				Departments:      deps,
				DepartmentsIDs:   depsIDs,
				MainDepartmentID: u.DepartmentGroup.URL,
				MainDepartment:   fmt.Sprintf("%s (%s)", u.DepartmentGroup.Name, u.DepartmentGroup.URL),
				Heads:            heads,
			})
			if err != nil {
				return err
			}
		}

		uri = strings.TrimSpace(result.Links.Next)
	}

	return nil
}

func (c *Client) WalkABCMembers(ctx context.Context, fn func(user string) error, services ...string) error {
	query := url.Values{
		"_limit":                       {"500"},
		"_fields":                      {"person.login"},
		"person.official.is_dismissed": {"false"},
		"person.official.affiliation":  {"yandex"},
		"person.official.is_robot":     {"false"},
		"group.url":                    {strings.Join(services, ",")},
	}
	uri := fmt.Sprintf("%s/v3/groupmembership?%s", strings.TrimRight(c.upstream, "/"), query.Encode())

	for uri != "" {
		var errResult staffErrorResponse
		var result staffGroupMembershipResponse
		rsp, err := c.httpc.R().
			SetContext(ctx).
			SetResult(&result).
			SetError(&errResult).
			ForceContentType("application/json").
			Get(uri)

		if err != nil {
			return fmt.Errorf("request failed while request %q: %w", uri, err)
		}

		if errResult.Msg != "" {
			return fmt.Errorf("remote error on request %q: %s", uri, errResult.Msg)
		}

		if !rsp.IsSuccess() {
			return fmt.Errorf("request failed: non-200 status code for request %q: %s", uri, rsp.Status())
		}

		for _, r := range result.Result {
			err = fn(r.Person.Login)
			if err != nil {
				return err
			}
		}

		uri = strings.TrimSpace(result.Links.Next)
	}

	return nil
}
