package main

import (
	"errors"
	"fmt"
	"os"
	"path/filepath"
	"strings"

	"github.com/mitchellh/go-homedir"
	"github.com/spf13/pflag"

	"a.yandex-team.ru/security/libs/go/sectools"
	"a.yandex-team.ru/security/skotty/launcher/internal/installer"
	"a.yandex-team.ru/security/skotty/launcher/internal/launcher"
	"a.yandex-team.ru/security/skotty/launcher/internal/logger"
	"a.yandex-team.ru/security/skotty/launcher/internal/updater"
	"a.yandex-team.ru/security/skotty/launcher/internal/version"
	"a.yandex-team.ru/security/skotty/libs/isroot"
)

var excludedArgs = []string{
	"-self-install-dir", "--self-install-dir",
}

func fatalf(format string, args ...interface{}) {
	_, _ = fmt.Fprintf(os.Stderr, "skotty-launcher: "+format+"\n", args...)
	os.Exit(1)
}

func main() {
	var launcherInstallDir, setChannel string
	var doUpdate, doHashUpdate, doShowLauncherVersion, doInstallLauncher, doHelp, doIsLauncher, allowRoot bool

	flags := pflag.NewFlagSet("skotty-launcher", pflag.ContinueOnError)
	flags.StringVar(&setChannel, "set-channel", "", "set distribution channel (stable/testing)")
	flags.BoolVar(&doUpdate, "update", false, "updates skotty")
	flags.BoolVar(&doHashUpdate, "has-update", false, "checks skotty update")
	flags.BoolVar(&doShowLauncherVersion, "self-version", false, "prints launcher version")
	flags.BoolVar(&doInstallLauncher, "self-install", false, "install skotty launcher")
	flags.StringVar(&launcherInstallDir, "self-install-dir", installer.DefaultInstallDir, "use custom launcher install dir")
	flags.BoolVar(&doIsLauncher, "is-launcher", false, "check if this program is skotty launcher")
	flags.BoolVar(&doHelp, "self-help", false, "prints launcher help")
	flags.BoolVar(&allowRoot, "allow-root", false, "allow starts from root")

	_ = flags.MarkHidden("allow-root")
	// fake arg to suppress launcher help
	_ = flags.BoolP("help", "h", false, "")
	_ = flags.MarkHidden("help")
	flags.SortFlags = false
	flags.ParseErrorsWhitelist.UnknownFlags = true

	printHelp := func() {
		fmt.Println("Usage of skotty launcher:")
		fmt.Println(flags.FlagUsages())
	}

	if err := flags.Parse(os.Args); err != nil {
		_, _ = fmt.Fprintf(os.Stderr, "can't parse arguments: %v\n", err)
		printHelp()
		return
	}

	switch {
	case allowRoot:
		_ = os.Setenv("SKOTTY_ALLOW_ROOT", "yes")
	case isroot.IsRoot():
		fatalf("you're trying to launch Skotty from the privileged user, most likely this is not correct. Otherwise, use --allow-root flag")
	}

	if setChannel != "" {
		c, err := sectools.ParseChannel(setChannel)
		if err != nil {
			fatalf("can't parse channel: %v", err)
		}

		if err := updater.WriteChannel(c); err != nil {
			fatalf("%v", err)
		}

		fmt.Printf("channel changed to: %s\n", c)
		return
	}

	if doIsLauncher {
		fmt.Println("yep")
		return
	}

	if doHelp {
		printHelp()
		return
	}

	if doShowLauncherVersion {
		fmt.Println(version.Full())
		return
	}

	if doInstallLauncher {
		realInstallDir := os.Getenv("SKOTTY_INSTALL_DIR")
		if realInstallDir == "" {
			var err error
			realInstallDir, err = homedir.Expand(launcherInstallDir)
			if err != nil {
				fatalf("can't normalize install dir %q: %v", launcherInstallDir, err)
				return
			}

			realInstallDir, err = filepath.Abs(realInstallDir)
			if err != nil {
				fatalf("can't normalize install dir %q: %v", launcherInstallDir, err)
				return
			}

		}

		if err := installer.Install(realInstallDir); err != nil {
			fatalf("self-install failed with error: %v", err)
		}

		return
	}

	if doUpdate {
		latestVersion, err := updater.Update()
		if err != nil {
			if errors.Is(err, updater.ErrNoUpdates) {
				fmt.Println("No updates available. You are using the latest version.")
				return
			}

			fatalf("update fail: %v", err)
			return
		}

		if updater.IsSkottyStarted() {
			// restart only if skotty was already configured: SKOTTY-203

			logger.Info("restart skotty service")
			if err := launcher.Restart(); err != nil {
				logger.Warnf("unable to restart skotty service: %v", err)
			}
		}

		fmt.Printf("Updated to version: %s\n", latestVersion)
		return
	}

	if doHashUpdate {
		isLatest, latestVersion, err := updater.IsLatestVersion()
		if err != nil {
			fatalf("can't resolve latest version: %v", err)
			return
		}

		if isLatest {
			fmt.Println("No updates availabe. You are using the latest version.")
			return
		}

		fmt.Printf("New version available: %s\n", latestVersion)
		return
	}

	skottyArgs := func() []string {
		parsedArgs := make([]string, 0, len(os.Args)-1)
		for _, arg := range os.Args[1:] {
			if arg == "" {
				continue
			}

			if arg[0] == '-' {
				if idx := strings.IndexByte(arg, '='); idx > 0 {
					parsedArgs = append(parsedArgs, arg[:idx])
					parsedArgs = append(parsedArgs, arg[idx+1:])
					continue
				}
			}

			parsedArgs = append(parsedArgs, arg)
		}

		out := make([]string, 0, len(parsedArgs))
		for i := 0; i < len(parsedArgs); i++ {
			arg := parsedArgs[i]
			if arg[0] == '-' {
				excluded := false
				for _, exArg := range excludedArgs {
					if arg == exArg {
						excluded = true
						break
					}
				}

				if excluded {
					i++
					continue
				}
			}

			out = append(out, arg)
		}

		return out
	}

	_, err := launcher.Launch(skottyArgs()...)
	if err != nil {
		fatalf("can't launch: %v", err)
		return
	}
}
