package installer

import (
	"errors"
	"fmt"
	"os"
	"os/exec"
	"path/filepath"

	"a.yandex-team.ru/security/libs/go/ioatomic"
	"a.yandex-team.ru/security/skotty/launcher/internal/updater"
)

func Install(installDir string) error {
	selfExe, err := os.Executable()
	if err != nil {
		return fmt.Errorf("can't determine self executable: %w", err)
	}

	return install(selfExe, installDir)
}

func findParentDir(targetPath string) (string, error) {
	current := filepath.Clean(targetPath)
	for {
		if _, err := os.Stat(current); err == nil {
			return current, nil
		}

		next := filepath.Dir(current)
		if next == current {
			return "", errors.New("can't find parent dir to install")
		}

		current = next
	}
}

func install(selfExe, installDir string) error {
	parent, err := findParentDir(installDir)
	if err != nil {
		return err
	}

	if !havePerms(parent) {
		ok, err := raisePermissions(selfExe, installDir)
		if err != nil {
			return err
		}

		if !ok {
			return nil
		}
	}

	if err := os.MkdirAll(installDir, 0o755); err != nil {
		return fmt.Errorf("can't mk install dir: %w", err)
	}

	targetPath := filepath.Join(installDir, updater.BinaryName)
	if err := ioatomic.CopyFile(selfExe, targetPath); err != nil {
		return err
	}

	if err := updateEnv(installDir); err != nil {
		return err
	}

	envPath, err := exec.LookPath("skotty")
	switch {
	case err != nil:
		fmt.Printf("WARN!!! the skotty executable was not found in the current PATH. You probably need to reload shell or something.\n\n")
	case envPath != targetPath:
		fmt.Printf("WARN!!! unexpected skotty path, propably you need to configure env[PATH] manually: %s (expected) != %s (actual)\n\n", targetPath, envPath)
	}

	fmt.Printf("skotty launcher successfully installed into: %s\n", targetPath)
	return nil
}
