package installer

import (
	"fmt"
	"os"
	"strings"
	"syscall"
	"unsafe"

	"golang.org/x/sys/windows"
	"golang.org/x/sys/windows/registry"
)

const DefaultInstallDir = "~/.skotty/launcher"

func havePerms(_ string) bool {
	return true
}

func updateEnv(installDir string) error {
	getRegEnv := func(key string) (string, error) {
		k, err := registry.OpenKey(registry.CURRENT_USER, "Environment", registry.QUERY_VALUE)
		if err != nil {
			return "", err
		}
		defer func() { _ = k.Close() }()

		s, _, err := k.GetStringValue(key)
		return s, err
	}

	setRegEnv := func(key, value string) error {
		k, err := registry.OpenKey(registry.CURRENT_USER, "Environment", registry.SET_VALUE)
		if err != nil {
			return err
		}
		defer func() { _ = k.Close() }()

		return k.SetStringValue(key, value)
	}

	sendBroadcast := func() {
		const (
			hwndBroadcast          = windows.HWND(0xFFFF)
			wmSettingChange        = uint(0x001A)
			smtoAbortIsHung        = uint(0x0000)
			smtoNormal             = uint(0x0002)
			smtoNoTimeoutIfNotHung = uint(0x0008)
			timeout                = uint(5000)
		)

		var proc = syscall.NewLazyDLL("user32.dll").NewProc("SendMessageTimeoutW")
		envUTF, err := windows.UTF16PtrFromString("Environment")
		if err != nil {
			return
		}

		_, _, _ = proc.Call(
			uintptr(hwndBroadcast),
			uintptr(wmSettingChange),
			0,
			uintptr(unsafe.Pointer(envUTF)),
			uintptr(smtoNormal|smtoAbortIsHung|smtoNoTimeoutIfNotHung),
			uintptr(timeout),
			0,
		)
	}

	pathEnv, err := getRegEnv("PATH")
	if err != nil && !os.IsNotExist(err) {
		return fmt.Errorf("can't read HKCU\\Environment[PATH]: %w", err)
	}

	const envDelim = string(os.PathListSeparator)
	paths := strings.Split(pathEnv, envDelim)
	for _, p := range paths {
		if p == installDir {
			// launcher already installed
			return nil
		}
	}

	paths = append(paths, installDir)
	err = setRegEnv("PATH", strings.Join(paths, envDelim))

	if err != nil {
		return fmt.Errorf("can't update HKCU\\Environment[PATH]: %w", err)
	}

	sendBroadcast()
	return nil
}

func raisePermissions(_, installDir string) (bool, error) {
	return false, fmt.Errorf("not enough file permissions to skotty launcher into: %s", installDir)
}
