package updater

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"os"
	"os/exec"
	"path/filepath"
	"sync"
	"time"

	"github.com/mitchellh/go-homedir"

	"a.yandex-team.ru/security/libs/go/sectools"
	"a.yandex-team.ru/security/skotty/launcher/internal/logger"
	"a.yandex-team.ru/security/skotty/skotty/pkg/skottyctl"
)

const (
	DefaultChannel = sectools.ChannelStable
	DefaultVersion = "0.0.0"

	releasesPath = "~/.skotty/releases"
	ctlSockPath  = "~/.skotty/ctl.sock"
)

var (
	absReleasesPath     string
	absReleasesPathErr  error
	absReleasesPathOnce sync.Once

	sectoolsOpts []sectools.Option
	ErrNoUpdates = errors.New("no update available")
)

type ReleaseInfo struct {
	Version string
	Channel sectools.Channel
}

func (r ReleaseInfo) String() string {
	return fmt.Sprintf("%s@%s", r.Version, r.Channel)
}

func (r ReleaseInfo) IsZero() bool {
	return r.Version == "" || r.Version == DefaultVersion
}

func ReleasesPath() (string, error) {
	absReleasesPathOnce.Do(func() {
		absReleasesPath, absReleasesPathErr = homedir.Expand(releasesPath)
	})

	return absReleasesPath, absReleasesPathErr
}

func InReleasesPath(elem ...string) (string, error) {
	rp, err := ReleasesPath()
	if err != nil {
		return "", err
	}

	return filepath.Join(append([]string{rp}, elem...)...), nil
}

func ReadReleaseInfo() (ReleaseInfo, error) {
	out := ReleaseInfo{
		Version: DefaultVersion,
		Channel: DefaultChannel,
	}

	infoPath, err := InReleasesPath("current.json")
	if err != nil {
		return out, fmt.Errorf("get release path: %w", err)
	}

	channelInfoData, err := os.ReadFile(infoPath)
	if err != nil {
		return out, fmt.Errorf("read cur release info: %w", err)
	}

	if err = json.Unmarshal(channelInfoData, &out); err != nil {
		return out, fmt.Errorf("parse release info: %w", err)
	}

	if out.Channel == "" {
		out.Channel = sectools.ChannelStable
	}

	if out.IsZero() {
		out.Version = DefaultVersion
		return out, errors.New("no version in the release info")
	}

	return out, nil
}

func writeReleaseInfo(ri ReleaseInfo) error {
	if ri.IsZero() {
		return errors.New("no release info available")
	}

	targetPath, err := InReleasesPath("current.json")
	if err != nil {
		return err
	}

	releaseBytes, err := json.Marshal(ri)
	if err != nil {
		return err
	}

	return os.WriteFile(targetPath, releaseBytes, 0o644)
}

func sectoolsService() *sectools.Client {
	relInfo, _ := ReadReleaseInfo()

	opts := append(
		[]sectools.Option{
			sectools.WithPreferFastURL(),
			sectools.WithChannel(relInfo.Channel),
		},
		sectoolsOpts...,
	)

	return sectools.NewClient("skotty", opts...)
}

func IsLatestVersion() (bool, string, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()

	relInfo, err := ReadReleaseInfo()
	if err != nil || relInfo.IsZero() {
		latestVersion, err := sectoolsService().LatestVersion(ctx)
		return false, latestVersion, err
	}

	return sectoolsService().IsLatestVersion(ctx, relInfo.Version)
}

func WriteChannel(c sectools.Channel) error {
	ri, err := ReadReleaseInfo()
	if err != nil {
		return fmt.Errorf("unable to get current release info: %w", err)
	}
	ri.Channel = c
	return writeReleaseInfo(ri)
}

func Update() (string, error) {
	curRelease, _ := ReadReleaseInfo()

	logger.Info("request latest version")
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	latestVersion, err := sectoolsService().LatestVersion(ctx)
	if err != nil {
		return "", fmt.Errorf("can't determine latest version: %w", err)
	}

	logger.Infof("latest version is: %s", latestVersion)
	if latestVersion == curRelease.Version {
		return latestVersion, ErrNoUpdates
	}

	_, err = DownloadVersion(latestVersion)
	if err != nil {
		return "", err
	}

	runningVersion := func() string {
		if !IsSkottyStarted() {
			// SKOTTY-191
			return ""
		}

		sockPath, err := homedir.Expand(ctlSockPath)
		if err != nil {
			logger.Warnf("can't determine skotty ctl socket path: %v", err)
			return ""
		}

		sc, err := skottyctl.NewClient(sockPath)
		if err != nil {
			logger.Errorf("can't create skotty client: %v", err)
			return ""
		}
		defer func() { _ = sc.Close() }()

		ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer cancel()

		status, err := sc.Status(ctx)
		if err != nil {
			return ""
		}

		return status.Version
	}

	cleanUpReleases := func() error {
		relPath, err := ReleasesPath()
		if err != nil {
			return err
		}

		dirs, err := os.ReadDir(relPath)
		if err != nil {
			return err
		}

		excludedVersion := map[string]struct{}{
			latestVersion:      {},
			curRelease.Version: {},
		}
		if runningVer := runningVersion(); runningVer != "" {
			excludedVersion[runningVer] = struct{}{}
		}

		for _, dir := range dirs {
			if !dir.IsDir() {
				continue
			}

			dirName := dir.Name()
			if _, ok := excludedVersion[dirName]; ok {
				continue
			}

			if err := os.RemoveAll(filepath.Join(relPath, dirName)); err != nil {
				logger.Warnf("can't remove old release %q: %v", dirName, err)
				continue
			}

			logger.Infof("removed old release: %s", dirName)
		}

		return nil
	}

	logger.Info("clean up old release")
	if err := cleanUpReleases(); err != nil {
		logger.Warnf("fail: %v", err)
	}

	logger.Info("updating current release info")
	curRelease.Version = latestVersion
	if err := writeReleaseInfo(curRelease); err != nil {
		return "", err
	}

	return latestVersion, nil
}

func DownloadVersion(version string) (string, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Minute)
	defer cancel()

	logger.Infof("prepare release dir")
	targetPath, err := InReleasesPath(version, BinaryName)
	if err != nil {
		return "", err
	}

	err = os.MkdirAll(filepath.Dir(targetPath), 0o755)
	if err != nil {
		return "", err
	}

	targetFile, err := os.OpenFile(targetPath, os.O_RDWR|os.O_CREATE|os.O_TRUNC, 0o755)
	if err != nil {
		return "", err
	}
	defer func() { _ = targetFile.Close() }()

	logger.Info("downloading...")
	err = sectoolsService().DownloadVersion(ctx, version, targetFile)
	if err != nil {
		_ = os.RemoveAll(filepath.Dir(targetPath))
		return "", err
	}
	_ = targetFile.Close()

	logger.Infof("release downloaded into: %s", targetPath)
	cmd := exec.Command(targetPath, "notify", "install")
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	cmd.Env = append(os.Environ(), "UNDER_LAUNCHER=yes")
	if err := cmd.Run(); err != nil {
		return "", fmt.Errorf("can't run downloaded skotty: %w", err)
	}

	return targetPath, nil
}

func IsSkottyStarted() bool {
	sockPath, err := homedir.Expand(ctlSockPath)
	if err != nil {
		return false
	}

	_, err = os.Stat(sockPath)
	return err == nil
}
