package certutil

import (
	"crypto/ecdsa"
	"crypto/x509"
	"encoding/pem"
	"errors"
	"fmt"
)

func PemToCert(in []byte) (*x509.Certificate, error) {
	pemBlock, err := DecodePem(in, "CERTIFICATE")
	if err != nil {
		return nil, err
	}

	pub, err := x509.ParseCertificate(pemBlock.Bytes)
	if err != nil {
		return nil, fmt.Errorf("failed to parse pub key: %w", err)
	}

	return pub, nil
}

func PemToECPriv(in []byte) (*ecdsa.PrivateKey, error) {
	pemBlock, err := DecodePem(in, "EC PRIVATE KEY")
	if err != nil {
		return nil, err
	}

	privKey, err := x509.ParseECPrivateKey(pemBlock.Bytes)
	if err != nil {
		return nil, fmt.Errorf("failed to parse priv key: %w", err)
	}

	return privKey, nil
}

func CertToPem(cert *x509.Certificate) []byte {
	return pem.EncodeToMemory(&pem.Block{
		Type:  "CERTIFICATE",
		Bytes: cert.Raw,
	})
}

func DecodePem(in []byte, expectedType string) (*pem.Block, error) {
	pubBlock, _ := pem.Decode(in)
	if pubBlock == nil {
		return nil, errors.New("no PEM block decoded")
	}

	if expectedType != "" && pubBlock.Type != expectedType {
		return nil, fmt.Errorf("invalid key type: %s", pubBlock.Type)
	}

	return pubBlock, nil
}
