package certutil

import (
	"bytes"
	"crypto"
	"crypto/ecdsa"
	"crypto/rand"
	"crypto/rsa"
	"crypto/x509"
	"encoding/asn1"
	"fmt"
	"math/big"

	"golang.org/x/crypto/blake2b"
)

type ECDSASignature struct {
	R, S *big.Int
}

var errAsn1Decoding = fmt.Errorf("asn.1 decoding failed")

func Verify(cert *x509.Certificate, msg, sig []byte) (bool, error) {
	h, err := blake2b.New256(nil)
	if err != nil {
		return false, fmt.Errorf("can't create blake2 hasher: %w", err)
	}
	_, _ = h.Write(msg)
	digest := h.Sum(nil)

	switch pub := cert.PublicKey.(type) {
	case *rsa.PublicKey:
		err := rsa.VerifyPKCS1v15(pub, crypto.BLAKE2b_256, digest, sig)
		return err == nil, nil
	case *ecdsa.PublicKey:
		ecdsaSig, err := asn1decode(sig)
		if err != nil {
			return false, err
		}

		return ecdsa.Verify(pub, digest, ecdsaSig.R, ecdsaSig.S), nil
	default:
		return false, fmt.Errorf("unexpected key type: %T", pub)
	}
}

func Sign(keySigner crypto.Signer, msg []byte) ([]byte, error) {
	h, err := blake2b.New256(nil)
	if err != nil {
		return nil, fmt.Errorf("can't create blake2 hasher: %w", err)
	}
	_, _ = h.Write(msg)
	digest := h.Sum(nil)

	return keySigner.Sign(rand.Reader, digest, crypto.BLAKE2b_256)
}

func asn1decode(b []byte) (*ECDSASignature, error) {
	// parse the signature
	sig := new(ECDSASignature)
	_, err := asn1.Unmarshal(b, sig)
	if err != nil {
		return nil, errAsn1Decoding
	}
	// encode the signature again
	encoded, err := asn1.Marshal(*sig)
	if err != nil {
		return nil, errAsn1Decoding
	}
	if !bytes.Equal(b, encoded) {
		return nil, errAsn1Decoding
	}
	return sig, nil
}
