package netutil

import (
	"errors"
	"fmt"
	"net"
	"syscall"

	"golang.org/x/sys/unix"
)

func UnixSocketCreds(c net.Conn) (UnixCreds, error) {
	unixConn, ok := c.(*net.UnixConn)
	if !ok {
		return UnixCreds{
			UID: -1,
			PID: -1,
		}, nil
	}

	raw, err := unixConn.SyscallConn()
	if err != nil {
		return UnixCreds{}, fmt.Errorf("conn SyscallConn: %w", err)
	}

	var cred *unix.Xucred
	var pid int
	cerr := raw.Control(func(fd uintptr) {
		cred, err = unix.GetsockoptXucred(int(fd), unix.SOL_LOCAL, unix.LOCAL_PEERCRED)
		if err != nil {
			err = fmt.Errorf("unix.GetsockoptXucred: %w", err)
			return
		}

		pid, err = unix.GetsockoptInt(int(fd), unix.SOL_LOCAL, unix.LOCAL_PEERPID)
		if err != nil {
			// that's fine
			if errors.Is(err, syscall.ENOTCONN) {
				err = nil
				pid = -1
			} else {
				err = fmt.Errorf("unix.GetsockoptInt: %w", err)
			}
		}
	})
	if cerr != nil {
		return UnixCreds{}, fmt.Errorf("raw.Control: %w", cerr)
	}
	if err != nil {
		return UnixCreds{}, err
	}

	return UnixCreds{
		UID: int(cred.Uid),
		PID: pid,
	}, nil
}
