package netutil

import (
	"fmt"
	"net"

	"golang.org/x/sys/unix"
)

func UnixSocketCreds(c net.Conn) (UnixCreds, error) {
	unixConn, ok := c.(*net.UnixConn)
	if !ok {
		return UnixCreds{
			UID: -1,
			PID: -1,
		}, nil
	}

	raw, err := unixConn.SyscallConn()
	if err != nil {
		return UnixCreds{}, fmt.Errorf("conn SyscallConn: %w", err)
	}

	var cred *unix.Xucred
	cerr := raw.Control(func(fd uintptr) {
		cred, err = unix.GetsockoptXucred(int(fd), unix.SOL_LOCAL, unix.LOCAL_PEERCRED)
	})
	if cerr != nil {
		return UnixCreds{}, fmt.Errorf("raw.Control: %w", cerr)
	}
	if err != nil {
		return UnixCreds{}, fmt.Errorf("unix.GetsockoptXucred: %w", err)
	}

	return UnixCreds{
		UID: int(cred.Uid),
		// TODO(buglloc): seems FreeBSD 13+ have cr_pid
		PID: -1,
	}, nil
}
