package skotty

import (
	"encoding/json"
	"fmt"

	"gopkg.in/yaml.v2"
)

var _ yaml.Unmarshaler = (*CertType)(nil)
var _ yaml.Marshaler = (*CertType)(nil)
var _ json.Marshaler = (*CertType)(nil)
var _ json.Unmarshaler = (*CertType)(nil)

type CertType uint8

const (
	CertTypeNone CertType = iota
	CertTypeSudo
	CertTypeSecure
	CertTypeInsecure
	CertTypeLegacy
	CertTypeRenew
)

func (c CertType) String() string {
	switch c {
	case CertTypeSecure:
		return "secure"
	case CertTypeInsecure:
		return "insecure"
	case CertTypeSudo:
		return "sudo"
	case CertTypeLegacy:
		return "legacy"
	case CertTypeRenew:
		return "renew"
	case CertTypeNone:
		return "none"
	default:
		return fmt.Sprintf("cert_type_%d", c)
	}
}

func (c *CertType) FromString(s string) error {
	switch s {
	case "secure":
		*c = CertTypeSecure
	case "insecure":
		*c = CertTypeInsecure
	case "sudo":
		*c = CertTypeSudo
	case "legacy":
		*c = CertTypeLegacy
	case "renew":
		*c = CertTypeRenew
	case "none":
		*c = CertTypeNone
	default:
		return fmt.Errorf("unknown cert type: %s", s)
	}

	return nil
}

func (c CertType) MarshalJSON() ([]byte, error) {
	return json.Marshal(c.String())
}

func (c *CertType) UnmarshalJSON(in []byte) error {
	var s string
	if err := json.Unmarshal(in, &s); err != nil {
		return err
	}

	return c.FromString(s)
}

func (c CertType) MarshalYAML() (interface{}, error) {
	return c.String(), nil
}

func (c *CertType) UnmarshalYAML(unmarshal func(interface{}) error) error {
	var s string
	if err := unmarshal(&s); err != nil {
		return err
	}

	return c.FromString(s)
}
