package skotty

import (
	"context"
	"fmt"
)

type EnrollClient struct {
	httpc *HTTPClient
}

func NewEnrollClient(opts ...EnrollOption) *EnrollClient {
	httpOpts := make([]Option, 0, len(opts))
	for _, o := range opts {
		ho, ok := o.(Option)
		if !ok {
			continue
		}
		httpOpts = append(httpOpts, ho)
	}

	return &EnrollClient{
		httpc: NewHTTPClient(httpOpts...),
	}
}

func (c *EnrollClient) RequestEnrollment(ctx context.Context, req *RequestEnrollmentReq) (*RequestEnrollmentRsp, error) {
	var out RequestEnrollmentRsp
	var serviceErr ServiceError
	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetBody(req).
		SetError(&serviceErr).
		SetResult(&out).
		ForceContentType("application/json").
		Put("/api/v1/host/enroll")

	if err != nil {
		return nil, fmt.Errorf("request failed: %w", err)
	}

	if !rsp.IsSuccess() {
		if serviceErr.Code != ServiceErrorCodeNone {
			return nil, &serviceErr
		}

		return nil, fmt.Errorf("request failed: non-200 status code: %s", rsp.Status())
	}

	return &out, nil
}

func (c *EnrollClient) IssueEnrollment(ctx context.Context, enrollID string, req *IssueEnrollmentReq) (*IssueEnrollmentRsp, error) {
	var out IssueEnrollmentRsp
	var serviceErr ServiceError
	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetBody(req).
		SetError(&serviceErr).
		SetResult(&out).
		ForceContentType("application/json").
		SetPathParam("enrollID", enrollID).
		Post("/api/v1/host/enroll/{enrollID}/issue")

	if err != nil {
		return nil, fmt.Errorf("request failed: %w", err)
	}

	if !rsp.IsSuccess() {
		if serviceErr.Code != ServiceErrorCodeNone {
			return nil, &serviceErr
		}

		return nil, fmt.Errorf("request failed: non-200 status code: %s", rsp.Status())
	}

	return &out, nil
}

func (c *EnrollClient) RequestRenewing(ctx context.Context, req *RequestRenewReq) (*RequestRenewRsp, error) {
	var out RequestRenewRsp
	var serviceErr ServiceError
	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetBody(req).
		SetError(&serviceErr).
		SetResult(&out).
		ForceContentType("application/json").
		Put("/api/v1/host/renew")

	if err != nil {
		return nil, fmt.Errorf("request failed: %w", err)
	}

	if !rsp.IsSuccess() {
		if serviceErr.Code != ServiceErrorCodeNone {
			return nil, &serviceErr
		}

		return nil, fmt.Errorf("request failed: non-200 status code: %s", rsp.Status())
	}

	return &out, nil
}

func (c *EnrollClient) ApproveRenew(ctx context.Context, enrollID string, req *ApproveRenewReq) error {
	var serviceErr ServiceError
	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetBody(req).
		SetError(&serviceErr).
		ForceContentType("application/json").
		SetPathParam("enrollID", enrollID).
		Post("/api/v1/host/renew/{enrollID}/approve")

	if err != nil {
		return fmt.Errorf("request failed: %w", err)
	}

	if !rsp.IsSuccess() {
		if serviceErr.Code != ServiceErrorCodeNone {
			return &serviceErr
		}

		return fmt.Errorf("request failed: non-200 status code: %s", rsp.Status())
	}

	return nil
}

func (c *EnrollClient) IssueRenew(ctx context.Context, enrollID string, req *IssueRenewReq) (*IssueRenewRsp, error) {
	var out IssueRenewRsp
	var serviceErr ServiceError
	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetBody(req).
		SetError(&serviceErr).
		SetResult(&out).
		ForceContentType("application/json").
		SetPathParam("enrollID", enrollID).
		Post("/api/v1/host/renew/{enrollID}/issue")

	if err != nil {
		return nil, fmt.Errorf("request failed: %w", err)
	}

	if !rsp.IsSuccess() {
		if serviceErr.Code != ServiceErrorCodeNone {
			return nil, &serviceErr
		}

		return nil, fmt.Errorf("request failed: non-200 status code: %s", rsp.Status())
	}

	return &out, nil
}

func (c *EnrollClient) RequestPubs(ctx context.Context) (*GetPubsRsp, error) {
	var out GetPubsRsp
	var serviceErr ServiceError

	rsp, err := c.httpc.R().
		SetContext(ctx).
		SetError(&serviceErr).
		SetResult(&out).
		ForceContentType("application/json").
		Get("/api/v1/ca/pubs")

	if err != nil {
		return nil, fmt.Errorf("request failed: %w", err)
	}

	if !rsp.IsSuccess() {
		if serviceErr.Code != ServiceErrorCodeNone {
			return nil, &serviceErr
		}

		return nil, fmt.Errorf("request failed: non-200 status code: %s", rsp.Status())
	}

	return &out, nil
}
