package sshagent

import (
	"golang.org/x/crypto/ssh"
	"golang.org/x/crypto/ssh/agent"

	"a.yandex-team.ru/library/go/core/log"
)

var _ agent.ExtendedAgent = (*LoggableHandler)(nil)

type LoggableHandler struct {
	Handler agent.ExtendedAgent
	Log     log.Logger
}

// List returns the identities known to the agent.
func (h *LoggableHandler) List() ([]*agent.Key, error) {
	h.Log.Debug(".List called")
	out, err := h.Handler.List()
	if err != nil {
		h.Log.Error(".List fail", log.Error(err))
	}

	return out, err
}

// Sign has the agent sign the data using a protocol 2 key as defined
// in [PROTOCOL.agent] section 2.6.2.
func (h *LoggableHandler) Sign(reqKey ssh.PublicKey, data []byte) (*ssh.Signature, error) {
	h.Log.Debug(".Sign called")
	out, err := h.Handler.Sign(reqKey, data)
	if err != nil {
		h.Log.Error(".Sign fail", log.Error(err))
	}

	return out, err
}

// SignWithFlags signs like Sign, but allows for additional flags to be sent/received
func (h *LoggableHandler) SignWithFlags(reqKey ssh.PublicKey, data []byte, flags agent.SignatureFlags) (*ssh.Signature, error) {
	h.Log.Debug(".SignWithFlags called")
	out, err := h.Handler.SignWithFlags(reqKey, data, flags)
	if err != nil {
		h.Log.Error(".SignWithFlags fail", log.Error(err))
	}

	return out, err
}

// Extension processes a custom extension request. Standard-compliant agents are not
// required to support any extensions, but this method allows agents to implement
// vendor-specific methods or add experimental features. See [PROTOCOL.agent] section 4.7.
func (h *LoggableHandler) Extension(extensionType string, contents []byte) ([]byte, error) {
	h.Log.Debug(".Extension called")
	out, err := h.Handler.Extension(extensionType, contents)
	if err != nil && err != agent.ErrExtensionUnsupported {
		h.Log.Error(".Extension fail",
			log.String("extension", extensionType),
			log.Error(err),
		)
	}

	return out, err
}

// Signers returns signers for all the known keys.
func (h *LoggableHandler) Signers() ([]ssh.Signer, error) {
	h.Log.Debug(".Signers called")
	out, err := h.Handler.Signers()
	if err != nil {
		h.Log.Error(".Signers fail", log.Error(err))
	}

	return out, err
}

// Add adds a private key to the agent.
func (h *LoggableHandler) Add(newKey agent.AddedKey) error {
	h.Log.Debug(".Add called")
	err := h.Handler.Add(newKey)
	if err != nil {
		h.Log.Error(".Add fail", log.String("key_name", newKey.Comment), log.Error(err))
	}

	return err
}

// Remove removes all identities with the given public key.
func (h *LoggableHandler) Remove(reqKey ssh.PublicKey) error {
	h.Log.Debug(".Remove called")
	err := h.Handler.Remove(reqKey)
	if err != nil {
		h.Log.Error(".Remove fail", log.Error(err))
	}

	return err
}

// RemoveAll removes all identities.
func (h *LoggableHandler) RemoveAll() error {
	h.Log.Debug(".RemoveAll called")
	err := h.Handler.RemoveAll()
	if err != nil {
		h.Log.Error(".RemoveAll fail", log.Error(err))
	}

	return err
}

// Lock locks the agent. Sign and Remove will fail, and List will empty an empty list.
func (h *LoggableHandler) Lock(passphrase []byte) error {
	h.Log.Debug(".Lock called")
	err := h.Handler.Lock(passphrase)
	if err != nil {
		h.Log.Error(".Lock fail", log.Error(err))
	}

	return err
}

// Unlock undoes the effect of Lock
func (h *LoggableHandler) Unlock(passphrase []byte) error {
	h.Log.Debug(".Unlock called")
	err := h.Handler.Unlock(passphrase)
	if err != nil {
		h.Log.Error(".Unlock fail", log.Error(err))
	}

	return err
}
