package winres

import (
	"fmt"
	"image"
	"io"

	"github.com/tc-hib/winres"
	"github.com/tc-hib/winres/version"
)

type ProgramInfo struct {
	Version     string
	FileVersion string
	Name        string
	Logo        io.Reader
}

func GenTo(pi ProgramInfo, dst io.Writer) error {
	rs := &winres.ResourceSet{}

	vi, err := pi.VersionInfo()
	if err != nil {
		return fmt.Errorf("create version info: %w", err)
	}

	rs.SetVersionInfo(vi)
	if pi.Logo != nil {
		img, _, err := image.Decode(pi.Logo)
		if err != nil {
			return fmt.Errorf("decode logo: %w", err)
		}

		icon, err := winres.NewIconFromResizedImage(img, nil)
		if err != nil {
			return fmt.Errorf("create logo: %w", err)
		}

		if err := rs.SetIcon(winres.ID(1), icon); err != nil {
			return fmt.Errorf("set logo: %w", err)
		}
	}

	return rs.WriteObject(dst, winres.ArchAMD64)
}

func (p ProgramInfo) VersionInfo() (version.Info, error) {
	type field struct {
		Name  string
		Value string
	}

	fields := []field{
		{
			Name:  version.LegalCopyright,
			Value: "© Yandex LLC",
		},
		{
			Name:  version.ProductName,
			Value: p.Name,
		},
	}

	var vi version.Info
	vi.SetProductVersion(p.Version)
	if p.FileVersion != "" {
		vi.SetFileVersion(p.FileVersion)
	} else {
		vi.SetFileVersion(p.Version)
	}

	for _, f := range fields {
		if err := vi.Set(version.LangDefault, f.Name, f.Value); err != nil {
			return version.Info{}, fmt.Errorf("failed to set field %q: %w", f.Name, err)
		}
	}

	return vi, nil
}
