package main

import (
	"errors"
	"fmt"
	"os"
	"runtime"

	"github.com/spf13/pflag"

	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/robossh/internal/logger"
	"a.yandex-team.ru/security/skotty/robossh/internal/osutil"
	"a.yandex-team.ru/security/skotty/robossh/internal/version"
)

func fatalf(msg string, a ...interface{}) {
	_, _ = fmt.Fprintf(os.Stderr, "robossh-agent: "+msg+"\n", a...)
	os.Exit(1)
}

func main() {
	runtime.GOMAXPROCS(1)
	if err := osutil.DisableDumps(); err != nil {
		fatalf("%v", err)
	}

	var (
		killFlag     bool
		debugMode    bool
		printVersion bool
		startFlag    startFlags
	)

	fs := pflag.NewFlagSet(os.Args[0], pflag.ContinueOnError)
	fs.StringVarP(&startFlag.bindAddress, "bind-address", "a", "", "Bind the agent to the UNIX-domain socket bind_address. The default is $TMPDIR/robossh-XXXXXXXXXX/agent.<ppid>.")
	fs.BoolVarP(&startFlag.foregroundMode, "foreground", "D", false, "Foreground mode. When this option is specified robossh-agent will not fork.")
	fs.BoolVar(&startFlag.withCerts, "with-certs", false, "Issue SSH certificates. Needed Skotty OAuth token in the env[ROBOSSH_TOKEN].")
	fs.StringSliceVar(&startFlag.certTypes, "request-ca", []string{skotty.CertTypeInsecure.String()}, "required certificate authorities (currently supported 'secure' and 'insecure')")
	fs.StringVarP(&startFlag.ttl, "ttl", "t", "", "Set a default value for the maximum lifetime of identities added to the agent. The lifetime may be specified in seconds or in a time format specified in sshd_config(5).")
	fs.BoolVarP(&debugMode, "debug", "d", false, "Debug mode. When this option is specified robossh-agent will not fork and will write debug information to standard error.")
	fs.BoolVarP(&killFlag, "kill", "k", false, "Kill the current agent (given by the SSH_AGENT_PID environment variable).")
	fs.BoolVar(&printVersion, "version", false, "Print robossh-agent version.")

	fs.SortFlags = false
	if err := fs.Parse(os.Args[1:]); err != nil {
		if errors.Is(err, pflag.ErrHelp) {
			return
		}
		fatalf("parse flags: %v", err)
	}

	startFlag.nArgs = fs.Args()
	startFlag.foregroundMode = startFlag.foregroundMode || debugMode

	if err := logger.InitLogger(debugMode); err != nil {
		fatalf("setup logger: %v", err)
	}

	if printVersion {
		version.WriteFullInfo(os.Stdout)
		return
	}

	if killFlag {
		if err := doKill(); err != nil {
			fatalf("cannot kill agent: %v", err)
		}
		return
	}

	if err := doStart(startFlag); err != nil {
		fatalf("%v", err)
	}
}
