package main

import (
	"errors"
	"fmt"
	"os"
)

type syncFlags struct {
	toAgent bool
	toDir   string
}

func doSync(flags syncFlags) error {
	token, ok := os.LookupEnv("ROBOSSH_TOKEN")
	if !ok {
		return errors.New("robossh-sync requires env[ROBOSSH_TOKEN], but it's not provided")
	}

	if token == "" {
		return errors.New("robossh-sync requires env[ROBOSSH_TOKEN], but it's empty")
	}

	if !flags.toAgent && flags.toDir == "" {
		return errors.New("you must provide where robossh-sync must sync certificates: --agent or --dir /path/to/.ssh")
	}

	if flags.toDir != "" {
		if err := checkDir(flags.toDir); err != nil {
			return err
		}
	}

	syncer := newCertSyncer()
	certs, err := syncer.IssueCertificates()
	if err != nil {
		return fmt.Errorf("unable to issue certificates: %w", err)
	}

	if flags.toAgent {
		socketPath := os.Getenv("SSH_AUTH_SOCK")
		if socketPath == "" {
			return errors.New("requested sync to authenticated agent, but no SSH_AUTH_SOCK provided")
		}

		if err := syncer.SyncAgent(socketPath, certs...); err != nil {
			return fmt.Errorf("authenticated agent sync fail: %w", err)
		}
	}

	if flags.toDir != "" {
		if err := syncer.SyncDir(flags.toDir, certs...); err != nil {
			return fmt.Errorf("dir sync fail: %w", err)
		}
	}

	return nil
}

func checkDir(p string) error {
	dirStat, err := os.Stat(p)
	if err != nil {
		if !os.IsNotExist(err) {
			return fmt.Errorf("unable to stat destination dir: %w", err)
		}

		if err := os.MkdirAll(p, 0o700); err != nil {
			return fmt.Errorf("unable to create destination dir: %w", err)
		}
		return nil
	}

	if !dirStat.IsDir() {
		return fmt.Errorf("destination dir is not a directory: %s", p)
	}

	return nil
}
