package main

import (
	"errors"
	"fmt"
	"os"
	"runtime"

	"github.com/spf13/pflag"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/robossh/internal/issuer"
	"a.yandex-team.ru/security/skotty/robossh/internal/logger"
	"a.yandex-team.ru/security/skotty/robossh/internal/osutil"
	"a.yandex-team.ru/security/skotty/robossh/internal/syncer"
	"a.yandex-team.ru/security/skotty/robossh/internal/version"
)

func fatalf(msg string, a ...interface{}) {
	_, _ = fmt.Fprintf(os.Stderr, "robossh-syncer: "+msg+"\n", a...)
	os.Exit(1)
}

func main() {
	runtime.GOMAXPROCS(1)
	if err := osutil.DisableDumps(); err != nil {
		fatalf("%v", err)
	}

	var (
		debugMode    bool
		printVersion bool
		syncFlag     syncFlags
	)

	fs := pflag.NewFlagSet(os.Args[0], pflag.ContinueOnError)
	fs.BoolVar(&syncFlag.toAgent, "agent", false, "Sync issued certificates to the authentication agent. The authentication agent must be running and the SSH_AUTH_SOCK environment variable must contain the name of its socket")
	fs.StringVar(&syncFlag.toDir, "dir", "", "Sync issued certificates to the directory")
	fs.BoolVarP(&debugMode, "debug", "d", false, "Debug mode.")
	fs.BoolVar(&printVersion, "version", false, "Print robossh-sync version.")
	fs.SortFlags = false

	if err := fs.Parse(os.Args[1:]); err != nil {
		if errors.Is(err, pflag.ErrHelp) {
			return
		}
		fatalf("parse flags: %v", err)
	}

	if err := logger.InitLogger(debugMode); err != nil {
		fatalf("setup logger: %v", err)
	}

	if printVersion {
		version.WriteFullInfo(os.Stdout)
		return
	}

	if err := doSync(syncFlag); err != nil {
		fatalf("%v", err)
	}
}

func newCertSyncer() *syncer.CertSyncer {
	robocOpts := []skotty.RoboOption{
		skotty.WithClientVersion(version.Full()),
		skotty.WithRoboAuthToken(os.Getenv("ROBOSSH_TOKEN")),
	}

	if upstream := os.Getenv("ROBOSSH_UPSTREAM"); upstream != "" {
		logger.Info("will use custom upstream", log.String("upstream", upstream))
		robocOpts = append(robocOpts, skotty.WithUpstream(upstream))
	}

	certsIssuer := issuer.NewSkottyIssuer(
		issuer.WithSkottyRoboClient(skotty.NewRoboClient(robocOpts...)),
	)

	return syncer.NewCertSyncer(certsIssuer)
}
