package issuer

import (
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"crypto/x509"
	"crypto/x509/pkix"
	"fmt"
	"math/big"
	"time"

	"a.yandex-team.ru/security/skotty/libs/skotty"
)

func genCertificate(typ skotty.CertType) (*x509.Certificate, *ecdsa.PrivateKey, error) {
	csr := &x509.Certificate{
		SerialNumber: big.NewInt(int64(time.Now().Year())),
		Subject: pkix.Name{
			CommonName:         fmt.Sprintf("%s@robossh", typ),
			Country:            []string{"RU"},
			Province:           []string{"Moscow"},
			Locality:           []string{"Moscow"},
			Organization:       []string{"Yandex"},
			OrganizationalUnit: []string{"Infra"},
		},
		NotBefore:             time.Now(),
		NotAfter:              time.Now().AddDate(25, 0, 0),
		ExtKeyUsage:           []x509.ExtKeyUsage{x509.ExtKeyUsageClientAuth},
		KeyUsage:              x509.KeyUsageDigitalSignature | x509.KeyUsageCertSign,
		BasicConstraintsValid: true,
	}

	privKey, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
	if err != nil {
		return nil, nil, fmt.Errorf("failed to generate private key: %w", err)
	}

	pubKeyBytes, err := x509.CreateCertificate(rand.Reader, csr, csr, &privKey.PublicKey, privKey)
	if err != nil {
		return nil, nil, fmt.Errorf("failed to generate certificate: %w", err)
	}

	cert, err := x509.ParseCertificate(pubKeyBytes)
	if err != nil {
		return nil, nil, fmt.Errorf("failed to parse issued certificate: %w", err)
	}

	return cert, privKey, nil
}
