package issuer

import (
	"context"
	"fmt"
	"math/rand"
	"os"
	"time"

	"golang.org/x/crypto/ssh"
	"golang.org/x/crypto/ssh/agent"

	"a.yandex-team.ru/security/skotty/libs/certutil"
	"a.yandex-team.ru/security/skotty/libs/skotty"
)

var _ Issuer = (*SkottyIssuer)(nil)

type SkottyIssuer struct {
	hostname  string
	serial    string
	roboc     *skotty.RoboClient
	certTypes []skotty.CertType
}

func NewSkottyIssuer(opts ...SkottyIssuerOption) *SkottyIssuer {
	hostname, err := os.Hostname()
	if err != nil {
		hostname = "n/a"
	}

	serial := fmt.Sprintf("%d-%d",
		os.Getpid(),
		rand.New(rand.NewSource(time.Now().UnixNano())).Intn(999)+1000,
	)

	issuer := &SkottyIssuer{
		hostname:  hostname,
		serial:    serial,
		roboc:     skotty.NewRoboClient(),
		certTypes: []skotty.CertType{skotty.CertTypeInsecure},
	}

	for _, opt := range opts {
		opt(issuer)
	}

	return issuer
}

func (s *SkottyIssuer) Certificates(ctx context.Context) ([]agent.AddedKey, error) {
	reqCerts := make([]skotty.RequestedCertificate, len(s.certTypes))
	out := make([]agent.AddedKey, len(s.certTypes))

	for i, ct := range s.certTypes {
		cert, priv, err := genCertificate(ct)
		if err != nil {
			return nil, fmt.Errorf("generate cert of type %s", ct)
		}

		reqCerts[i] = skotty.RequestedCertificate{
			Type:   ct,
			HostID: i,
			Cert:   certutil.CertToPem(cert),
		}

		out[i] = agent.AddedKey{
			PrivateKey: priv,
			Comment:    fmt.Sprintf("%s@robossh", ct),
		}
	}

	rsp, err := s.roboc.IssueCertificates(ctx, &skotty.IssueRoboCertificatesReq{
		Hostname:     s.hostname,
		TokenSerial:  s.serial,
		TokenName:    fmt.Sprintf("robossh[%d] on %s", os.Getpid(), s.hostname),
		Certificates: reqCerts,
	})
	if err != nil {
		return nil, fmt.Errorf("request enrollment service: %w", err)
	}

	for _, cert := range rsp.Certificates {
		sshPub, _, _, _, err := ssh.ParseAuthorizedKey(cert.SSHCert)
		if err != nil {
			return nil, fmt.Errorf("failed to parse ssh pub key %q: %w", cert.Type, err)
		}

		sshCert, ok := sshPub.(*ssh.Certificate)
		if !ok {
			return nil, fmt.Errorf("service returned invalid SSH cert %q: %T", cert.Type, sshPub)
		}

		out[cert.HostID].Certificate = sshCert
	}

	return out, nil
}
