package keystore

import (
	"sync"

	"a.yandex-team.ru/security/skotty/robossh/internal/agentkey"
)

type Store struct {
	keys []*agentkey.Key
	mu   sync.Mutex
}

func NewStore() *Store {
	return &Store{}
}

func (s *Store) Len() int {
	s.mu.Lock()
	defer s.mu.Unlock()

	return len(s.keys)
}

func (s *Store) Get(fp string) (*agentkey.Key, bool) {
	s.mu.Lock()
	defer s.mu.Unlock()

	for _, key := range s.keys {
		if key.Fingerprint() == fp {
			return key, true
		}
	}

	return nil, false
}

func (s *Store) Range(fn func(*agentkey.Key) bool) {
	s.mu.Lock()
	defer s.mu.Unlock()

	for _, key := range s.keys {
		if !fn(key) {
			break
		}
	}
}

func (s *Store) Add(key *agentkey.Key) bool {
	if k, ok := s.Get(key.Fingerprint()); ok {
		// already exists
		k.Update(key)
		return false
	}

	s.mu.Lock()
	defer s.mu.Unlock()

	s.keys = append(s.keys, key)
	return true
}

func (s *Store) Remove(fp string) bool {
	s.mu.Lock()
	defer s.mu.Unlock()

	for i, key := range s.keys {
		if key.Fingerprint() != fp {
			continue
		}

		s.keys[i] = s.keys[len(s.keys)-1]
		s.keys = s.keys[:len(s.keys)-1]
		return true
	}

	return false
}

func (s *Store) RemoveAll() {
	s.mu.Lock()
	defer s.mu.Unlock()

	s.keys = s.keys[:0]
}

func (s *Store) RemoveExpired() int {
	s.mu.Lock()
	defer s.mu.Unlock()

	last := len(s.keys)
	var c, i int
	for i < last {
		if !s.keys[i].IsExpired() {
			i++
			continue
		}

		c++
		last--
		s.keys[i] = s.keys[last]
		s.keys = s.keys[:last]
	}

	return c
}
