package logger

import (
	"os"

	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/library/go/core/log"
	xzap "a.yandex-team.ru/library/go/core/log/zap"
)

var L log.Logger

func InitLogger(debug bool) error {
	logLvl := zap.InfoLevel
	if debug {
		logLvl = zap.DebugLevel
	}

	output := "stderr"
	if newOutput := os.Getenv("ROBOSSH_LOG_PATH"); newOutput != "" {
		output = newOutput
	}

	if fi, err := os.Stat(output); err == nil && fi.Mode()&os.ModeType == 0 {
		_ = os.Truncate(output, 0)
	}

	zlog, err := xzap.New(zap.Config{
		Level:            zap.NewAtomicLevelAt(logLvl),
		Encoding:         "console",
		OutputPaths:      []string{output},
		ErrorOutputPaths: []string{"stderr"},
		InitialFields: map[string]interface{}{
			"pid": os.Getpid(),
		},
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:     "msg",
			LevelKey:       "level",
			TimeKey:        "ts",
			CallerKey:      "caller",
			NameKey:        "name",
			EncodeLevel:    zapcore.CapitalLevelEncoder,
			EncodeTime:     zapcore.ISO8601TimeEncoder,
			EncodeDuration: zapcore.StringDurationEncoder,
			EncodeCaller:   zapcore.ShortCallerEncoder,
		},
	})
	if err != nil {
		return err
	}

	L = zlog.AddCallerSkip(1)
	return nil
}

func Named(name string) log.Logger {
	return L.WithName(name)
}

func Debug(msg string, fields ...log.Field) {
	L.Debug(msg, fields...)
}

func Info(msg string, fields ...log.Field) {
	L.Info(msg, fields...)
}

func Warn(msg string, fields ...log.Field) {
	L.Warn(msg, fields...)
}

func Error(msg string, fields ...log.Field) {
	L.Error(msg, fields...)
}

func Fatal(msg string, fields ...log.Field) {
	L.Fatal(msg, fields...)
}

func Structured() log.Structured {
	return L.Structured()
}
