package main

import (
	"encoding/base64"
	"flag"
	"fmt"
	"log"
	"os"
	"path/filepath"
	"strings"
)

var keys = []string{
	"secure", "insecure", "sudo", "renew",
}

func fatalf(msg string, a ...interface{}) {
	_, _ = fmt.Fprintf(os.Stderr, "key_unpacker: "+msg+"\n", a...)
	os.Exit(1)
}

func main() {
	var outDir string
	flag.StringVar(&outDir, "out", ".", "output dir")
	flag.Parse()

	prevVer := os.Getenv("SKOTTY_PREV_VER")
	curVer := os.Getenv("SKOTTY_CUR_VER")
	nextVer := os.Getenv("SKOTTY_NEXT_VER")
	if curVer == "" {
		fatalf("empty env[SKOTTY_CUR_VER]")
	}
	if nextVer == "" {
		fatalf("empty env[SKOTTY_NEXT_VER]")
	}
	if curVer == nextVer {
		fatalf("same env[SKOTTY_CUR_VER] (%s) and env[SKOTTY_NEXT_VER] (%s)", curVer, nextVer)
	}

	if err := os.MkdirAll(outDir, 0o700); err != nil {
		fatalf("can't create keys dir: %v", err)
	}

	type verInfo struct {
		Ver          string
		TargetSuffix string
	}
	versions := []verInfo{
		{
			Ver:          curVer,
			TargetSuffix: "cur.json",
		},
		{
			Ver:          nextVer,
			TargetSuffix: "next.json",
		},
	}
	if prevVer != "" {
		versions = append(versions, verInfo{
			Ver:          prevVer,
			TargetSuffix: "prev.json",
		})
	}

	for _, k := range keys {
		for _, v := range versions {
			envKey := fmt.Sprintf("CA_%s_%s", strings.ToUpper(k), v.Ver)
			blob := os.Getenv(envKey)
			if blob == "" {
				fatalf("empty env[%s]", envKey)
			}

			data, err := base64.StdEncoding.DecodeString(blob)
			if err != nil {
				fatalf("invalid env[%s] key blob: %v", envKey, err)
			}

			targetFile := filepath.Join(outDir, fmt.Sprintf("%s_%s", strings.ToLower(k), v.TargetSuffix))
			err = os.WriteFile(targetFile, data, 0o600)
			if err != nil {
				fatalf("failed to write file %s: %v", targetFile, err)
			}

			log.Printf("wrote file %s from blob %s\n", targetFile, envKey)
		}
	}
}
