import React, {useState, useEffect, useCallback} from "react";
import './Authorize.scss';
import cn from 'bem-cn-lite';
import { decode as base64_decode} from 'js-base64';
import {Button, Loader, HelpTooltip, TextInput, toaster} from '@yandex-data-ui/common';
import {useSkottyApi} from '../../services';
import {AuthInfo} from '../../models/skotty';
import {BigError} from '../BigError/BigError';

const b = cn('authorization');

export default function Authorize() {
    const skottyApi = useSkottyApi();
    const authId = window.SK.authId;
  
    const parsedHash = new URLSearchParams(window.location.hash.substr(1));
    let authToken = parsedHash.get('token');
  
    const [errorMsg, setErrorMsg] = useState<string>();
    const [done, setDone] = useState(false);
    const [authInfo, setAuthInfo] = useState<AuthInfo | undefined>(undefined);
    const [loading, setLoading] = useState(true);

    const [formErrorMsg, setFormErrorMsg] = useState<string>();
    const [formLoading, setFormLoading] = useState(false);
    const [mfaCode, setMfaCode] = useState<string>();
    const [codeSended, setCodeSended] = useState(false);

    const showFormError = function(error: string) {
      toaster?.createToast({
        name: "ooops",
        title: "ooops",
        type: "error",
        content: error,
      });
    }

    if (!!authToken && authToken.startsWith("atv2-")) {
      authToken = base64_decode(authToken.substring(5));
    }

    useEffect(() => {
      if (!authId?.match(/^[0-9a-z_-]+$/)) {
        setErrorMsg(`invalid auth id: ${authId}`);
        return;
      }

      if (!authToken) {
        setErrorMsg('no auth token provided');
        return;
      }

      skottyApi
        .fetchAuthInfo(authId, {authToken})
        .then(r => {
          if (!!r.errorCode) {
            setErrorMsg(r.errorMsg);
            return;
          }
  
          setAuthInfo(r);
          setLoading(false);
        })
        .catch((error: { message: any; }) => {
          setErrorMsg(`fetch failed: ${error.message}`);
        });
    }, [skottyApi, authId, authToken]);
    
    const sendCode = useCallback(function() {
      if (!authId?.match(/^[0-9a-z_-]+$/)) {
        setFormErrorMsg(`invalid auth id: ${authId}`);
        return;
      }

      if (!authToken) {
        setFormErrorMsg('shit happens');
        return;
      }

      const req = {
        authToken: authToken,
      };

      setFormLoading(true);
      skottyApi
        .sendCode(authId, req)
        .then(r => {
          setFormLoading(false);

          if (!!r.errorCode) {
            showFormError(r.errorMsg);
            return;
          }
  
          setCodeSended(true);          
        })
        .catch((error: { message: any; }) => {
          setFormLoading(false);
          showFormError(error.message);
        });
    }, [skottyApi, authId, authToken]);

    const authorize = useCallback(function() {
      if (!authId?.match(/^[0-9a-z_-]+$/)) {
        setFormErrorMsg(`invalid auth id: ${authId}`);
        return;
      }

      if (!mfaCode || !/^[0-9]{6}$/.test(mfaCode)) {
        return;
      }

      if (!authToken) {
        setFormErrorMsg('shit happens');
        return;
      }

      const req = {
        authToken: authToken,
        mfaCode: Number(mfaCode),
      };
      
      setFormLoading(true);
      skottyApi
        .authApprove(authId, req)
        .then(r => {
          setFormLoading(false);

          if (!!r.errorCode) {
            showFormError(r.errorMsg);
            return;
          }
  
          setDone(true);
        })
        .catch((error: { message: any; }) => {
          setFormLoading(false);
          showFormError(error.message);
        });
    }, [skottyApi, authId, authToken, mfaCode]);
    
    const renderErrorMsg = function(msg: string) {
      if (msg === 'already authorized') {
        return (
          <div className={b('center_card')}>
            <p>You have already approved this authorization.</p>
            <p><a href="/">your tokens</a></p>
          </div>
        )
      }

      return <BigError msg={msg}/>;
    }

    const renderInfoTooltip = function() {
      const renderTokenTooltip = function() {
        return <>
          type: {authInfo?.tokenType}<br/>
          id: {authInfo?.tokenId}
        </>;
      }

      return <HelpTooltip size="l" content={renderTokenTooltip()} offset={{top: -10}}/>
    }

    const mfaCodeKeydownHandler = useCallback(function(e: KeyboardEvent) {
        if (e.key !== 'Enter') {
          return;
        }

        e.stopPropagation();
        e.preventDefault();
        
        if (codeSended) {
          authorize();
          return;
        } else {
          sendCode();
          return;
        }
    }, [codeSended, authorize, sendCode]);

    useEffect(() => {
      if (done) {
        return;
      }

      setTimeout(() => {        
        window.addEventListener('keydown', mfaCodeKeydownHandler);
      }, 0);

      return () => window.removeEventListener('keydown', mfaCodeKeydownHandler);;
    }, [done, mfaCodeKeydownHandler]);

    return (
      <div className={b()}>
        <div className={b('inner')}>
        {(done && <span>OK, return to terminal</span>) || 
        (errorMsg && renderErrorMsg(errorMsg)) ||
        (loading && <Loader size="l"/>) ||
          (
            <React.Fragment>
              <div className={b('card')}>
                <div>To issue certificates for your token "{authInfo?.tokenName}"{renderInfoTooltip()}, we need to make sure that you are you ;)</div>
                {
                  (!codeSended && <div>So we need to send a sms with a one-time code to your phone {authInfo?.phone}</div>) ||
                  (
                    <div>
                      We have just sent sms to you, now you have to enter code from sms to this form:
                      <div className={b('code_form')}>
                        <TextInput autoFocus={true} size="l" placeholder="123456" onUpdate={setMfaCode} error={mfaCode && !/^[0-9]{6}$/.test(mfaCode) && "should be 6 digits"}/>
                      </div>
                    </div>
                  )
                }
              </div>
              <div>
                {
                  (!codeSended && <Button view="action" size="l" onClick={sendCode} loading={formLoading}>Let's do this</Button>) ||
                  <Button view="action" size="l" onClick={authorize} loading={formLoading}>Confirm</Button>
                }
              </div>
              {formErrorMsg && <p><span className={b('error_text')}>ooops: {formErrorMsg}</span></p>}
            </React.Fragment>
          )
        }
        </div>
      </div>
    );
}
