import React, {useEffect, useState, useCallback} from "react";
import './TokenView.scss';
import cn from 'bem-cn-lite';
import Moment from "moment";
import {Link, Icon, Loader, Table, Tabs, TabsDirection, toaster} from '@yandex-data-ui/common';
import faDownload from '@yandex-data-ui/fontawesome/svgs/solid/download.svg';
import {AuditMsg, Certificate, CertType, Token} from "../../models/skotty";
import {TokenState} from '../TokenState/TokenState';
import {CertState} from '../CertState/CertState';
import {RevokeButton} from '../RevokeButton/RevokeButton';
import {useSkottyApi} from "../../services";
import {TokenState as ETS} from '../../models/skotty';

const b = cn('token_view');

const TAB_OVERVIEW = 'overview';
const TAB_AUDIT_LOG = 'audit_log';
const TAB_USAGE_HISTORY = 'usage_history';

interface TokenViewProps {
    className?: string;
    loading: boolean;
    token: Token | undefined;
}

let certsCache: {[index: string]:Certificate[]} = {}

function showToastErr(error: string) {
    toaster?.createToast({
      name: "ooops",
      title: "ooops",
      type: "error",
      content: error,
    });
}

export function TokenView(props: React.PropsWithChildren<TokenViewProps>) {
    const skottyApi = useSkottyApi();
    const [certs, setCerts] = useState<Certificate[]>([]);
    const [auditMsgs, setAuditMsgs] = useState<AuditMsg[]>([]);
    const [errorMsg, setErrorMsg] = useState("");
    const [activeTab, setActiveTab] = useState(TAB_OVERVIEW);
    const [tokenState, setTokenState] = useState(ETS.None);
    const [revoking, setRevoking] = useState(false);

    useEffect(() => {
        if (!props.token) {
            return;
        }

        setTokenState(props.token?.tokenState);
    }, [props.token])

    useEffect(() => {
        if (!props.token || activeTab != TAB_OVERVIEW) {
            return;
        }

        const cacheKey = `${props.token.user}|${props.token.id}|${props.token.enrollId}`;
        if (cacheKey in certsCache) {
            setCerts(certsCache[cacheKey]);
            return;
        }

        setErrorMsg("");
        skottyApi
            .listCerts(props.token.user, props.token.id, props.token.enrollId)
            .then(r => {
                if (!!r.errorCode) {
                    setErrorMsg(r.errorMsg);
                    return;
                }

                setCerts(r);
                certsCache[cacheKey] = r;
            })
            .catch(error => {
                setErrorMsg(`fetch failed: ${error.message}`);
            });
    }, [skottyApi, props.token, activeTab]);

    useEffect(() => {
        if (!props.token || activeTab != TAB_AUDIT_LOG) {
            return;
        }

        setErrorMsg("");
        skottyApi
            .listAuditMsgs(props.token.user, props.token.id, props.token.enrollId)
            .then(r => {
                if (!!r.errorCode) {
                    setErrorMsg(r.errorMsg);
                    return;
                }

                setAuditMsgs(r);
            })
            .catch(error => {
                setErrorMsg(`fetch failed: ${error.message}`);
            });
    }, [skottyApi, props.token, activeTab]);

    const revokeToken = useCallback(() => {
        if (!props.token) {
            return;
        }
        
        setRevoking(true);
        skottyApi
            .revoke(props.token.user, props.token.id, props.token.enrollId)
            .then(r => {
                if (!!r.errorCode) {
                    showToastErr(r.errorMsg);
                    return;
                }

                setRevoking(false);
                setTokenState(ETS.Revoking);
            })
            .catch(error => {
                setRevoking(false);
                showToastErr(`fetch failed: ${error.message}`);
            });
    }, [skottyApi, props.token]);

    const renderTokenInfo = (token: Token) => {
        const downloads = function(cert: Certificate) {
            if (cert.certType == CertType.Renew) {
                return <React.Fragment>
                    <span className={b('ca_inactive')}><Icon data={faDownload} size={16}/>&nbsp;x509</span>
                    &nbsp;
                    <span className={b('ca_inactive')}><Icon data={faDownload} size={16}/>&nbsp;ssh</span>
                </React.Fragment>
            }

            const downloadCertPrefix = `/api/v1/front/token/${encodeURIComponent(token.user)}/${encodeURIComponent(token.id)}/${encodeURIComponent(token.enrollId)}/cert`;

            return (
                <React.Fragment>
                    <Link href={`${downloadCertPrefix}/${cert.serial}/skotty-${cert.serial}-cert.pem`} target="_blank"><Icon data={faDownload} size={16}/>&nbsp;x509</Link>
                    &nbsp;
                    <Link href={`${downloadCertPrefix}/${cert.serial}/skotty-${cert.serial}.pub`} target="_blank"><Icon data={faDownload} size={16}/>&nbsp;ssh</Link>
                </React.Fragment>
            )
        }

        return (
            <React.Fragment>
                <div className={b('info_spacer')}>
                    <div>Token "{token.name}"</div>
                    <div style={{marginLeft: '10px'}}><TokenState state={tokenState} expires={token.expiresAt} /></div>
                    <div><RevokeButton tokenName={token.name} onConfirm={revokeToken} loading={revoking} disabled={tokenState != ETS.Active}/></div>
                </div>
                <div style={{display: 'flex'}}>
                    <div className={b('token_info')}>
                        <div>
                            <div>Created at: {Moment(token.createdAt*1000).format("DD.MM.YYYY HH:mm")}</div>
                            <div>Updated at: {Moment(token.updatedAt*1000).format("DD.MM.YYYY HH:mm")}</div>
                            <div>Expires: {Moment(token.expiresAt*1000).format("DD.MM.YYYY HH:mm")}</div>
                        </div>
                        <div style={{width: '20px'}}/>
                        <div>
                            <div>Type: {token.tokenType}</div>
                            <div>ID: <code>{token.id}</code></div>
                            <div>Enrollment ID: <code>{token.enrollId}</code></div>
                        </div>
                    </div>
                </div>

                <div className={b('info_spacer')}>
                    Certificates/Keys
                </div>
                <Table
                    columns={[
                        {
                            id: 'serial',
                            name: 'Serial #'
                        },
                        {
                            id: 'certType',
                            name: 'Type'
                        },
                        {
                            id: 'certState',
                            name: 'State',
                            template: (cert: Certificate) => {
                                return <CertState state={cert.certState} expires={cert.validBefore} />
                            }
                        },
                        {
                            align: 'right',
                            id: 'createdAt',
                            name: 'Created',
                            template: (cert: Certificate) => {
                                return <div>{Moment(cert.createdAt*1000).format("DD.MM.YYYY HH:mm")}</div>
                            }
                        },
                        {
                            align: 'right',
                            id: 'validBefore',
                            name: 'Expires',
                            template: (cert: Certificate) => {
                                if (cert.validBefore === 253402300799) {
                                    return <React.Fragment>∞</React.Fragment>
                                }
                                return <div>{Moment(cert.validBefore*1000).format("DD.MM.YYYY HH:mm")}</div>
                            }
                        },
                        {
                            id: 'fingerprint',
                            name: 'Fingerprint'
                        },
                        {
                            align: 'right',
                            id: 'pubs',
                            name: 'Pubs',
                            template: downloads,
                        }
                    ]}
                    data={certs}
                    emptyMessage={errorMsg || "No data available ¯\_(ツ)_/¯"}
                    stickyHorizontalScrollBreakpoint={0}
                />
            </React.Fragment>
        )
    }

    const renderAuditLog = (msgs: AuditMsg[]) => {
        return (
            <React.Fragment>
                <Table
                    columns={[
                        {
                            id: 'ts',
                            name: 'Time',
                            template: (cert: AuditMsg) => {
                                return <div>{Moment(cert.ts*1000).format("DD.MM.YYYY HH:mm:ss")}</div>
                            }
                        },
                        {
                            id: 'message',
                            name: 'Message',
                            template: (cert: AuditMsg) => {
                                return <code>{cert.message}</code>
                            }
                        },
                    ]}
                    data={msgs}
                    emptyMessage={errorMsg || "No data available ¯\_(ツ)_/¯"}
                    stickyHorizontalScrollBreakpoint={0}
                />
            </React.Fragment>
        )
    }

    const renderActiveTab = () => {
        if (!props.token) {
            return <div></div>;
        }

        if (props.loading) {
            return <div className={b('loader')}><Loader size="m" /></div>;
        }

        switch (activeTab) {
            case TAB_OVERVIEW:
                return renderTokenInfo(props.token);
            case TAB_AUDIT_LOG:
                return renderAuditLog(auditMsgs);
            case TAB_USAGE_HISTORY:
                return <p>TODO</p>;
            default:
                return <p>shit happens</p>;
        }
    }

    return (
        <div className={b()}>
            <Tabs
                className={b('tabs', props.className)}
                direction={TabsDirection.Horizontal}
                items={[
                    {
                        id: TAB_OVERVIEW,
                        title: 'Overview',
                        disabled: props.loading,
                    },
                    {
                        id: TAB_AUDIT_LOG,
                        title: 'Audit log',
                        disabled: props.loading,
                    },
                    {
                        id: TAB_USAGE_HISTORY,
                        title: 'Usage history (comming soon)',
                        disabled: true,
                    }
                ]}
                activeTab={activeTab}
                onSelectTab={(tabId) => setActiveTab(tabId)}
            />
            <div className={b('content')}>
                {renderActiveTab()}
            </div>
        </div>
    );
}
