import {CancelToken, AxiosResponse, AxiosError} from 'axios';
import _ from 'lodash';
import {RequestService, RequestConfig, ServiceError} from './request-service';
import {Token, Certificate, AuthInfo, AuditMsg} from '../models/skotty';

export type AuthInfoReq = {
    authToken: string;
}

export type AuthApproveReq = {
    authToken: string;
    mfaCode: number;
}

export type SendCodeReq = {
    authToken: string;
}

export class SkottyApi extends RequestService {
    request<R = unknown>(
        {method = 'GET', url, data}: RequestConfig,
        cancelToken?: CancelToken,
    ) {

        return this.axios
            .request({
                method: method,
                url: `/api/v1/front/${url}`,
                data,
                cancelToken,
            })
            .then((response: AxiosResponse<R & ServiceError>) => response.data)
            .catch((error: AxiosError) => {
                if (this.isCancel(error)) {
                    console.info('[skotty] Request canceled');
                }

                throw error;
            });
    }

    listTokens(login: string): Promise<Token[] & ServiceError> {
        return this.request<Token[]>({
            method: 'get',
            url: `token/${encodeURIComponent(login)}`,
        });
    }

    fetchAuthInfo(id: string, req: AuthInfoReq): Promise<AuthInfo & ServiceError> {
        return this.request<AuthInfo>({
            method: 'post',
            url: `auth/${encodeURIComponent(id)}/info`,
            data: req,
        });
    }

    authApprove(id: string, req: AuthApproveReq): Promise<ServiceError> {
        return this.request({
            method: 'post',
            url: `auth/${encodeURIComponent(id)}/approve`,
            data: req,
        });
    }

    sendCode(id: string, req: SendCodeReq): Promise<ServiceError> {
        return this.request({
            method: 'post',
            url: `auth/${encodeURIComponent(id)}/send-code`,
            data: req,
        });
    }

    revoke(login: string, tokenId: string, enrollId: string): Promise<ServiceError> {
        return this.request({
            method: 'delete',
            url: `token/${encodeURIComponent(login)}/${encodeURIComponent(tokenId)}/${encodeURIComponent(enrollId)}`,
        });
    }

    listCerts(login: string, tokenId: string, enrollId: string): Promise<Certificate[] & ServiceError> {
        return this.request<Certificate[]>({
            method: 'get',
            url: `token/${encodeURIComponent(login)}/${encodeURIComponent(tokenId)}/${encodeURIComponent(enrollId)}/certs`,
        });
    }

    listAuditMsgs(login: string, tokenId: string, enrollId: string): Promise<AuditMsg[] & ServiceError> {
        return this.request<AuditMsg[]>({
            method: 'get',
            url: `token/${encodeURIComponent(login)}/${encodeURIComponent(tokenId)}/${encodeURIComponent(enrollId)}/audit.log`,
        });
    }
}
