package app

import (
	"context"
	"fmt"
	"net/http"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/httputil/middleware/recovery"
	"a.yandex-team.ru/security/libs/go/chim"
	"a.yandex-team.ru/security/skotty/service/internal/app/controller"
	"a.yandex-team.ru/security/skotty/service/internal/app/controller/admapi"
	"a.yandex-team.ru/security/skotty/service/internal/app/controller/capi"
	"a.yandex-team.ru/security/skotty/service/internal/app/controller/common"
	"a.yandex-team.ru/security/skotty/service/internal/app/controller/frontapi"
	"a.yandex-team.ru/security/skotty/service/internal/app/controller/hostapi"
	"a.yandex-team.ru/security/skotty/service/internal/app/controller/roboapi"
	"a.yandex-team.ru/security/skotty/service/internal/app/env"
	"a.yandex-team.ru/security/skotty/service/internal/logger"
)

type App struct {
	env   *env.Env
	http  *http.Server
	ctrls []controller.Controller
}

func NewApp(e *env.Env) (*App, error) {
	out := &App{
		env:  e,
		http: new(http.Server),
	}

	return out, out.initialize()
}

func (a *App) initialize() error {
	r := chi.NewRouter()
	r.Use(recovery.New(
		recovery.WithLogger(logger.Structured()),
		recovery.WithCallBack(func(w http.ResponseWriter, _ *http.Request, e error) {
			w.WriteHeader(http.StatusInternalServerError)
			_, _ = w.Write([]byte(fmt.Sprintf("ship happens: %s", e)))
		}),
	))
	r.Use(chim.RealIP)
	r.Use(chim.LogRequest(a.env.Log))

	controllers := map[string]func(e *env.Env) (controller.Controller, error){
		"/v1/front":  frontapi.NewController,
		"/v1/host":   hostapi.NewController,
		"/v1/ca":     capi.NewController,
		"/v1/admin":  admapi.NewController,
		"/v1/robots": roboapi.NewController,
		"/common":    common.NewController,
	}

	for pattern, f := range controllers {
		c, err := f(a.env)
		if err != nil {
			return fmt.Errorf("failed to create contoller %q: %w", pattern, err)
		}

		r.Route(pattern, func(r chi.Router) {
			c.BuildRoute(r)
		})

		a.ctrls = append(a.ctrls, c)
	}

	a.http.Handler = r
	return nil
}

func (a *App) ListenAndServe(addr string) error {
	logger.Info("start http API", log.String("addr", addr))
	a.http.Addr = addr
	return a.http.ListenAndServe()
}

func (a *App) Shutdown(ctx context.Context) error {
	return a.http.Shutdown(ctx)
}
