package admapi

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/skotty/service/internal/app/controller"
	"a.yandex-team.ru/security/skotty/service/internal/app/env"
	"a.yandex-team.ru/security/skotty/service/internal/models"
)

type Controller struct {
	*env.Env
}

func NewController(e *env.Env) (controller.Controller, error) {
	if e.Roles.Admin == "" {
		return nil, errors.New("no admin role provided")
	}

	return &Controller{
		Env: e,
	}, nil
}

func (c *Controller) BuildRoute(r chi.Router) {
	r.Use(authMiddleware(c.BlackBox, c.TVM))
	r.Use(checkACLMiddleware(c.TVM, c.Roles.Admin))

	r.Get("/roles", c.getRoles)
	r.Post("/revoke/{login}", c.revokeUser)
	r.Post("/revoke/{login}/{tokenID}", c.revokeUserToken)
}

func (c *Controller) getRoles(w http.ResponseWriter, r *http.Request) {
	roles, err := c.TVM.GetRoles(r.Context())
	if err != nil {
		c.Log.Error("failed to get TVM roles", log.Error(err))

		RespErrorf(w, http.StatusInternalServerError, "failed to get TVM roles: %v", err)
		return
	}

	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	w.WriteHeader(http.StatusOK)
	_, _ = w.Write(roles.GetRaw())
}

func (c *Controller) revokeUser(w http.ResponseWriter, r *http.Request) {
	login := chi.URLParam(r, "login")
	if login == "" || login == "*" {
		RespErrorf(w, http.StatusBadRequest, "invalid login: %q", login)
		return
	}

	revokedTokens, err := c.DB.ScheduleRevokeUserToken(r.Context(), login, "")
	if err != nil {
		c.Log.Error("failed to revoke token", log.Any("login", login), log.Error(err))

		RespErrorf(w, http.StatusInternalServerError, "failed to revoke token: %v", err)
		return
	}

	for _, token := range revokedTokens {
		c.AuditLog.Log(token.ID, token.EnrollID, "revoked by SIB")
	}

	out := make([]RevokedToken, len(revokedTokens))
	for i, t := range revokedTokens {
		out[i] = RevokedToken{
			User:     t.User,
			TokenID:  t.ID,
			EnrollID: t.EnrollID,
		}
	}

	c.Revoker.RevokeNow()
	RespOK(w, RevokeRsp{
		Tokens: out,
	})
}

func (c *Controller) revokeUserToken(w http.ResponseWriter, r *http.Request) {
	login := chi.URLParam(r, "login")
	if login == "" {
		RespErrorf(w, http.StatusBadRequest, "invalid login: %q", login)
		return
	}

	tokenID := chi.URLParam(r, "tokenID")
	if tokenID == "" || tokenID == "*" {
		RespErrorf(w, http.StatusBadRequest, "invalid tokenID: %q", login)
		return
	}

	var revokedTokens []models.TokenID
	var err error
	switch login {
	case "*":
		revokedTokens, err = c.DB.ScheduleRevokeTokenID(r.Context(), tokenID)
	default:
		revokedTokens, err = c.DB.ScheduleRevokeUserToken(r.Context(), login, tokenID)
	}

	if err != nil {
		c.Log.Error("failed to revoke token", log.Any("login", login), log.Any("token_id", login), log.Error(err))

		RespErrorf(w, http.StatusInternalServerError, "failed to revoke token: %v", err)
		return
	}

	for _, token := range revokedTokens {
		c.AuditLog.Log(token.ID, token.EnrollID, "revoked by SIB")
	}

	out := make([]RevokedToken, len(revokedTokens))
	for i, t := range revokedTokens {
		out[i] = RevokedToken{
			User:     t.User,
			TokenID:  t.ID,
			EnrollID: t.EnrollID,
		}
	}

	c.Revoker.RevokeNow()
	RespOK(w, RevokeRsp{
		Tokens: out,
	})
}

func (c *Controller) Shutdown(_ context.Context) {}

func RespOK(w http.ResponseWriter, rsp interface{}) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	w.WriteHeader(http.StatusOK)
	_ = json.NewEncoder(w).Encode(rsp)
}

func RespErrorf(w http.ResponseWriter, status int, msg string, a ...interface{}) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	w.WriteHeader(status)
	_ = json.NewEncoder(w).Encode(ErrorRsp{
		Code: status,
		Msg:  fmt.Sprintf(msg, a...),
	})
}
