package capi

import (
	"bytes"
	"context"
	"crypto/md5"
	"encoding/hex"
	"encoding/json"
	"fmt"
	"net/http"

	"github.com/go-chi/chi/v5"
	"golang.org/x/crypto/ssh"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/service/internal/app/controller"
	"a.yandex-team.ru/security/skotty/service/internal/app/env"
	"a.yandex-team.ru/security/skotty/service/internal/revoker"
)

type hostCa struct {
	Type  skotty.CertType `json:"type"`
	SSHCa []byte          `json:"ssh_ca"`
}

type hostGetPubsRsp struct {
	Keys []hostCa `json:"keys"`
}

type Controller struct {
	*env.Env
}

func NewController(e *env.Env) (controller.Controller, error) {
	return &Controller{
		Env: e,
	}, nil
}

func (c *Controller) BuildRoute(r chi.Router) {
	r.Get("/pubs", func(w http.ResponseWriter, r *http.Request) {
		authorities := c.CA.AllCAs()
		out := make([]hostCa, len(authorities))
		i := 0
		for typ, ca := range authorities {
			out[i] = hostCa{
				Type:  typ,
				SSHCa: ca.SSHPubBytes,
			}
			i++
		}

		w.Header().Set("Content-Type", "application/json; charset=utf-8")
		w.WriteHeader(http.StatusOK)
		_ = json.NewEncoder(w).Encode(hostGetPubsRsp{
			Keys: out,
		})
	})

	r.Route("/pub", func(r chi.Router) {
		newPubHandler := func(certTypes ...skotty.CertType) func(http.ResponseWriter, *http.Request) {
			content, sum := func() ([]byte, string) {
				var out bytes.Buffer
				for _, ct := range certTypes {
					ca, ok := c.CA.CA(ct)
					if !ok {
						c.Log.Error("can't find requested ca", log.String("ca", ct.String()))
						continue
					}

					out.Write(ca.TrustedSSHPub)
					out.WriteByte('\n')
				}

				content := out.Bytes()
				sum := md5.Sum(content)
				return content, fmt.Sprintf(`"%s"`, hex.EncodeToString(sum[:]))
			}()

			return func(w http.ResponseWriter, r *http.Request) {
				if ifNoneMatch := r.Header.Get("If-None-Match"); ifNoneMatch != "" {
					w.WriteHeader(http.StatusNotModified)
					return
				}

				w.Header().Set("Content-Type", "text/plain; charset=utf-8")
				w.Header().Set("Etag", sum)
				w.WriteHeader(http.StatusOK)
				_, _ = w.Write(content)
			}
		}

		r.Get("/ssh", newPubHandler(skotty.CertTypeInsecure, skotty.CertTypeSecure))
		r.Get("/sudo", newPubHandler(skotty.CertTypeSudo))
		r.Get("/insecure", newPubHandler(skotty.CertTypeInsecure))
		r.Get("/secure", newPubHandler(skotty.CertTypeSecure))
	})

	r.Get("/krl/all.zst", func(w http.ResponseWriter, r *http.Request) {
		err := c.Storage.Serve(r, revoker.KRLPath, w)
		if err != nil {
			w.Header().Set("Content-Type", "text/plain; charset=utf-8")
			w.WriteHeader(http.StatusInternalServerError)
			_, _ = w.Write([]byte(err.Error()))
		}
	})

	r.Get("/info", func(w http.ResponseWriter, r *http.Request) {
		var out bytes.Buffer
		authorities := c.CA.AllCAs()
		for ct, ca := range authorities {
			out.WriteString(fmt.Sprintf("%s (%s %s):\n", ct, ca.Public.Subject.CommonName, ca.SSHFingerprint))
			out.WriteString("-- trusted verifiers ---\n")
			for _, k := range ca.TrustedVerifiers {
				out.WriteString(k.Subject.CommonName)
				out.WriteByte('\n')
			}
			out.WriteString("-- KRL signers ---\n")
			for _, k := range ca.KRLSigners {
				out.Write(ssh.MarshalAuthorizedKey(k.PublicKey()))
			}
			out.WriteString("-- ssh pubs --\n")
			out.Write(ca.TrustedSSHPub)
			out.WriteByte('\n')
			out.WriteByte('\n')
		}

		w.Header().Set("Content-Type", "text/plain; charset=utf-8")
		w.WriteHeader(http.StatusOK)
		_, _ = w.Write(out.Bytes())
	})
}

func (c *Controller) Shutdown(_ context.Context) {}
