package frontapi

import (
	"context"
	"crypto/rand"
	"encoding/json"
	"fmt"
	"math/big"
	"net/http"
	"net/url"
	"path"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/masksecret"
	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/service/internal/app/controller"
	"a.yandex-team.ru/security/skotty/service/internal/app/env"
	"a.yandex-team.ru/security/skotty/service/internal/db"
	"a.yandex-team.ru/security/skotty/service/internal/models"
)

const (
	maxCodeTries = 10
	smsText      = "Skotty confirmation code: {{ code }}"
)

type Controller struct {
	*env.Env
}

func NewController(e *env.Env) (controller.Controller, error) {
	return &Controller{
		Env: e,
	}, nil
}

func (c *Controller) BuildRoute(r chi.Router) {
	r.Use(authMiddleware(c.BlackBox))
	r.Use(checkCsrfMiddleware(c.SignSecret))

	r.Group(func(r chi.Router) {
		roles := rolesMapping{
			Full: []string{c.Roles.Admin},
			Read: []string{c.Roles.Viewer},
		}
		r.Use(checkAccessMiddleware("login", roles, c.TVM))

		r.Get("/token/{login}", c.listUserTokens)
		r.Delete("/token/{login}/{tokenID}/{enrollID}", c.tokenRevoke)
		r.Get("/token/{login}/{tokenID}/{enrollID}/audit.log", c.tokenAuditLog)
		r.Get("/token/{login}/{tokenID}/{enrollID}/certs", c.listTokenCerts)
		r.Get("/token/{login}/{tokenID}/{enrollID}/cert/{serial}/{filename}", c.certPub)
	})

	r.Post("/auth/{authID}/info", c.authInfo)
	r.Post("/auth/{authID}/send-code", c.sendCode)
	r.Post("/auth/{authID}/approve", c.authApprove)
}

func (c *Controller) listUserTokens(w http.ResponseWriter, r *http.Request) {
	login := chi.URLParam(r, "login")
	if login == "" {
		RespErrorf(w, "empty login")
		return
	}

	tokens, err := c.DB.LookupUserTokens(r.Context(), login)
	if err != nil {
		RespErrorf(w, "can't find user tokens: %v", err)
		return
	}

	out := make([]Token, len(tokens))
	for i, token := range tokens {
		out[i] = Token(token)
	}

	RespOK(w, out)
}

func (c *Controller) tokenAuditLog(w http.ResponseWriter, r *http.Request) {
	login := chi.URLParam(r, "login")
	if login == "" {
		RespErrorf(w, "empty login")
		return
	}

	tokenID := chi.URLParam(r, "tokenID")
	if tokenID == "" {
		RespErrorf(w, "empty tokenID")
		return
	}

	enrollID := chi.URLParam(r, "enrollID")
	if enrollID == "" {
		RespErrorf(w, "empty enrollID")
		return
	}

	msgs, err := c.DB.LookupAuditMsgs(r.Context(), tokenID, enrollID)
	if err != nil {
		RespErrorf(w, "can't find audit messages: %v", err)
		return
	}

	out := make([]AuditMsg, len(msgs))
	for i, msg := range msgs {
		out[i] = AuditMsg{
			TS:      msg.TS / 1000000000,
			Message: msg.Message,
		}
	}

	RespOK(w, out)
}

func (c *Controller) tokenRevoke(w http.ResponseWriter, r *http.Request) {
	login := chi.URLParam(r, "login")
	if login == "" {
		RespErrorf(w, "empty login")
		return
	}

	tokenID := chi.URLParam(r, "tokenID")
	if tokenID == "" {
		RespErrorf(w, "empty tokenID")
		return
	}

	enrollID := chi.URLParam(r, "enrollID")
	if enrollID == "" {
		RespErrorf(w, "empty enrollID")
		return
	}

	revokedTokens, err := c.DB.ScheduleRevokeToken(r.Context(), login, tokenID, enrollID)
	if err != nil {
		c.Log.Error("failed to revoke token", log.Any("tfid", models.TFID(login, tokenID, enrollID)), log.Error(err))

		RespErrorf(w, "failed to revoke token: %v", err)
		return
	}

	if len(revokedTokens) == 0 {
		RespErrorf(w, "no suitable token found")
		return
	}

	for _, token := range revokedTokens {
		c.AuditLog.Log(token.ID, token.EnrollID, "revoked by user")
	}

	c.Revoker.RevokeNow()
	w.Header().Set("Content-Type", "application/json")
	_, _ = w.Write([]byte(`{}`))
}

func (c *Controller) listTokenCerts(w http.ResponseWriter, r *http.Request) {
	login := chi.URLParam(r, "login")
	if login == "" {
		RespErrorf(w, "empty login")
		return
	}

	tokenID := chi.URLParam(r, "tokenID")
	if tokenID == "" {
		RespErrorf(w, "empty tokenID")
		return
	}

	enrollID := chi.URLParam(r, "enrollID")
	if enrollID == "" {
		RespErrorf(w, "empty enrollID")
		return
	}

	certs, err := c.DB.LookupTokenCertsQuery(r.Context(), models.TFID(login, tokenID, enrollID))
	if err != nil {
		RespErrorf(w, "can't find token certs: %v", err)
		return
	}

	out := make([]Cert, len(certs))
	for i, cert := range certs {
		out[i] = Cert{
			Serial:        cert.Serial,
			CertType:      cert.CertType,
			CertState:     cert.CertState,
			CAFingerprint: cert.CAFingerprint,
			Fingerprint:   cert.SSHFingerprint,
			Principal:     cert.Principal,
			CreatedAt:     cert.CreatedAt,
			ValidAfter:    cert.ValidAfter,
			ValidBefore:   cert.ValidBefore,
		}
	}

	RespOK(w, out)
}

func (c *Controller) certPub(w http.ResponseWriter, r *http.Request) {
	login := chi.URLParam(r, "login")
	if login == "" {
		RespErrorf(w, "empty login")
		return
	}

	tokenID := chi.URLParam(r, "tokenID")
	if tokenID == "" {
		RespErrorf(w, "empty tokenID")
		return
	}

	enrollID := chi.URLParam(r, "enrollID")
	if enrollID == "" {
		RespErrorf(w, "empty enrollID")
		return
	}

	serial := chi.URLParam(r, "serial")
	if serial == "" {
		RespErrorf(w, "empty serial")
		return
	}

	filename := chi.URLParam(r, "filename")
	if filename == "" {
		RespErrorf(w, "empty filename")
		return
	}

	var out []byte
	tokenFullID := models.TFID(login, tokenID, enrollID)
	pubType := path.Ext(filename)
	switch pubType {
	case ".pem":
		pub, err := c.DB.LookupCertX509PubQuery(r.Context(), serial, tokenFullID)
		if err != nil {
			RespErrorf(w, "failed to retrieve certificate 'cert': %v", err)
			return
		}

		out = []byte(pub)
	case ".pub":
		pub, err := c.DB.LookupCertSSHPubQuery(r.Context(), serial, tokenFullID)
		if err != nil {
			RespErrorf(w, "failed to retrieve certificate 'ssh_cert': %v", err)
			return
		}

		out = []byte(pub)
	default:
		RespErrorf(w, "unsupported pub type: %s", pubType)
		return
	}

	w.Header().Set("Content-Type", "text/plain")
	w.Header().Set("Content-Disposition", fmt.Sprintf("attachment; filename=%s", url.QueryEscape(filename)))
	w.WriteHeader(http.StatusOK)
	_, _ = w.Write(out)
}

func (c *Controller) authInfo(w http.ResponseWriter, r *http.Request) {
	var req AuthInfoReq
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		RespErrorf(w, "failed to parse request: %v", err)
		return
	}

	authInfo, err := c.Auth.ParseAuthToken(req.AuthToken, models.AuthKindNone)
	if err != nil {
		RespErrorf(w, "invalid auth token: %v", err)
		return
	}

	user, ok := userFromContext(r.Context())
	if !ok {
		RespErrorf(w, "unknown user")
		return
	}

	if user.YandexUID == "" {
		RespErrorf(w, "no yandexuid")
		return
	}

	if authInfo.AuthID != chi.URLParam(r, "authID") {
		RespErrorf(w, "invalid auth id: %s != %s", authInfo.AuthID, chi.URLParam(r, "authID"))
		return
	}

	_, err = c.DB.LookupAuthorization(r.Context(), authInfo.AuthID)
	switch err {
	case db.ErrNotFound:
		// pass
	case nil:
		RespErrorf(w, "already authorized")
		return
	default:
		RespErrorf(w, "can't check auth state: %v", err)
		return
	}

	tokenID, err := skotty.SerialToTokenID(authInfo.TokenType, authInfo.TokenSerial)
	if err != nil {
		RespErrorf(w, "failed to generate token id: %v", err)
		return
	}

	phone, err := c.StaffAPI.UserSecurePhone(r.Context(), user.Login)
	if err != nil {
		RespErrorf(w, "can't get your main phone number from staff: %v", err)
		return
	}

	RespOK(w, AuthInfoRsp{
		AuthKind:  authInfo.AuthKind,
		TokenID:   tokenID,
		TokenType: authInfo.TokenType,
		TokenName: authInfo.TokenName,
		EnrollID:  authInfo.EnrollmentID,
		HostName:  authInfo.Hostname,
		Phone:     masksecret.Phone(phone),
	})
}

func (c *Controller) sendCode(w http.ResponseWriter, r *http.Request) {
	var req SendCodeReq
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		RespErrorf(w, "failed to parse request: %v", err)
		return
	}

	authInfo, err := c.Auth.ParseAuthToken(req.AuthToken, models.AuthKindNone)
	if err != nil {
		RespErrorf(w, "invalid auth token: %v", err)
		return
	}

	user, ok := userFromContext(r.Context())
	if !ok {
		RespErrorf(w, "unknown user")
		return
	}

	if user.YandexUID == "" {
		RespErrorf(w, "no yandexuid")
		return
	}

	if authInfo.AuthID != chi.URLParam(r, "authID") {
		RespErrorf(w, "invalid auth id: %s != %s", authInfo.AuthID, chi.URLParam(r, "authID"))
		return
	}

	otpCode, err := generateCode()
	if err != nil {
		RespErrorf(w, "can't generate one-time code: %v", err)
		return
	}

	phone, err := c.StaffAPI.UserSecurePhone(r.Context(), user.Login)
	if err != nil {
		RespErrorf(w, "can't get your main phone number from staff: %v", err)
		return
	}

	smsID, err := c.YaSMS.Send(r.Context(), phone, smsText, map[string]string{
		"code": fmt.Sprintf("%06d", otpCode),
	})
	if err != nil {
		RespErrorf(w, "failed to send SMS code: %v", err)
		return
	}

	err = c.DB.RequestMFA(r.Context(), authInfo.AuthID, user.Login, otpCode)
	if err != nil {
		RespErrorf(w, "failed to approve enrollment request: %v", err)
		return
	}

	c.Log.Info("webauth SMS sended", log.String("user", user.Login), log.String("id", smsID))
	w.Header().Set("Content-Type", "application/json")
	_, _ = w.Write([]byte(`{}`))
}

func (c *Controller) authApprove(w http.ResponseWriter, r *http.Request) {
	var req AuthApproveReq
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		RespErrorf(w, "failed to parse request: %v", err)
		return
	}

	authInfo, err := c.Auth.ParseAuthToken(req.AuthToken, models.AuthKindNone)
	if err != nil {
		RespErrorf(w, "invalid auth token: %v", err)
		return
	}

	user, ok := userFromContext(r.Context())
	if !ok {
		RespErrorf(w, "unknown user")
		return
	}

	if user.YandexUID == "" {
		RespErrorf(w, "no yandexuid")
		return
	}

	if authInfo.AuthID != chi.URLParam(r, "authID") {
		RespErrorf(w, "invalid auth id: %s != %s", authInfo.AuthID, chi.URLParam(r, "authID"))
		return
	}

	_, err = c.DB.LookupAuthorization(r.Context(), authInfo.AuthID)
	switch err {
	case db.ErrNotFound:
		// pass
	case nil:
		RespErrorf(w, "already authorized")
		return
	default:
		RespErrorf(w, "can't check auth state: %v", err)
		return
	}

	mfaInfo, err := c.DB.LookupMFA(r.Context(), authInfo.AuthID, user.Login)
	switch err {
	case db.ErrNotFound:
		RespErrorf(w, "authorization is already authorized or doesn't exists")
		return
	case nil:
		// pass
	default:
		RespErrorf(w, "can't check auth state: %v", err)
		return
	}

	if mfaInfo.Tries > maxCodeTries {
		RespErrorf(w, "tries exceed, you need to request a new code")
		return
	}

	if mfaInfo.Code != req.MFACode {
		if err := c.DB.UpdateMFATries(r.Context(), authInfo.AuthID, user.Login, mfaInfo.Tries+1); err != nil {
			c.Log.Error("can't update mfa tries",
				log.String("auth_id", authInfo.AuthID),
				log.String("user", user.Login),
				log.Error(err))
		}
		RespErrorf(w, "invalid MFA code")
		return
	}

	authorization := &models.Authorization{
		ID:         authInfo.AuthID,
		User:       user.Login,
		UserTicket: user.UserTicket,
	}
	authorization.Sign, err = c.Auth.SignAuthorization(authorization)
	if err != nil {
		c.Log.Error("authorization sign failed",
			log.String("auth_id", authInfo.AuthID),
			log.String("user", user.Login),
			log.Error(err))
		RespErrorf(w, "failed to sign your authorization approve: %v", err)
		return
	}

	err = c.DB.IssueAuthorization(r.Context(), authorization)
	if err != nil {
		RespErrorf(w, "failed to approve enrollment request: %v", err)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	_, _ = w.Write([]byte(`{}`))
}

func (c *Controller) Shutdown(_ context.Context) {}

func RespOK(w http.ResponseWriter, rsp interface{}) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	w.WriteHeader(http.StatusOK)
	_ = json.NewEncoder(w).Encode(rsp)
}

func RespErrorf(w http.ResponseWriter, msg string, a ...interface{}) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	w.WriteHeader(http.StatusOK)
	_ = json.NewEncoder(w).Encode(ErrorRsp{
		Code: 500,
		Msg:  fmt.Sprintf(msg, a...),
	})
}

func generateCode() (uint32, error) {
	nBig, err := rand.Int(rand.Reader, big.NewInt(999999))
	return uint32(nBig.Uint64()), err
}
