package hostapi

import (
	"bytes"
	"context"
	"encoding/base64"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"reflect"
	"strings"
	"time"

	"github.com/go-chi/chi/v5"
	"github.com/gofrs/uuid"
	"golang.org/x/crypto/ssh"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/skotty/libs/certutil"
	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/service/internal/app/controller"
	"a.yandex-team.ru/security/skotty/service/internal/app/env"
	"a.yandex-team.ru/security/skotty/service/internal/db"
	"a.yandex-team.ru/security/skotty/service/internal/models"
	"a.yandex-team.ru/security/skotty/service/internal/signer"
	"a.yandex-team.ru/security/skotty/service/internal/staff"
	"a.yandex-team.ru/security/skotty/service/internal/webauth"
	"a.yandex-team.ru/security/skotty/skotty/pkg/sshutil"
)

const (
	renewTTLGap = 24 * 31 * time.Hour
)

type Controller struct {
	*env.Env
}

func NewController(e *env.Env) (controller.Controller, error) {
	return &Controller{
		Env: e,
	}, nil
}

func (c *Controller) BuildRoute(r chi.Router) {
	r.Use(hostCheckAuthMiddleware)

	r.Put("/enroll", c.enrollRequest)
	r.Post("/enroll/{enrollID}/issue", c.enrollIssue)
	r.Put("/renew", c.renewRequest)
	r.Post("/renew/{enrollID}/approve", c.approveRenewIssue)
	r.Post("/renew/{enrollID}/issue", c.renewIssue)
}

func (c *Controller) enrollRequest(w http.ResponseWriter, r *http.Request) {
	var req skotty.RequestEnrollmentReq
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("failed to parse request: %v", err))
		return
	}

	if req.TokenSerial == "" {
		c.RespInvalidReq(w, "empty token serial")
		return
	}

	if req.Hostname == "" {
		c.RespInvalidReq(w, "empty hostname")
		return
	}

	rawEnrollID, err := uuid.NewV4()
	if err != nil {
		c.RespInternalError(w, fmt.Sprintf("can't generate enrollment ID: %v", err))
		return
	}

	rawAuthID, err := uuid.NewV4()
	if err != nil {
		c.RespInternalError(w, fmt.Sprintf("can't generate auth ID: %v", err))
		return
	}

	enrollID := rawEnrollID.String()
	authID := rawAuthID.String()
	hostToken, userToken, err := c.Auth.EnrollAuthTokens(webauth.AuthInfo{
		AuthID:       authID,
		TokenSerial:  req.TokenSerial,
		TokenType:    req.TokenType,
		TokenName:    req.TokenName,
		Hostname:     req.Hostname,
		EnrollmentID: enrollID,
	})
	if err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("can't generate auth token: %v", err))
		return
	}

	authURL := fmt.Sprintf(
		"https://%s/authorize/%s/#token=atv2-%s",
		r.Host, authID, base64.RawURLEncoding.EncodeToString([]byte(userToken)),
	)

	c.RespOK(w, skotty.RequestEnrollmentRsp{
		EnrollmentID: enrollID,
		AuthID:       authID,
		AuthToken:    hostToken,
		AuthURL:      authURL,
	})
}

func (c *Controller) renewRequest(w http.ResponseWriter, r *http.Request) {
	var req skotty.RequestRenewReq
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("failed to parse request: %v", err))
		return
	}

	if req.Hostname == "" {
		c.RespInvalidReq(w, "empty hostname")
		return
	}

	var renewInfo *RenewInfo
	if req.RenewToken != "" {
		ri, err := c.Auth.ParseRenew(req.RenewToken)
		if err != nil {
			c.RespInvalidReq(w, fmt.Sprintf("failed to parse renew token: %v", err))
			return
		}

		renewInfo = &RenewInfo{
			User:         ri.User,
			TokenSerial:  ri.TokenSerial,
			TokenType:    ri.TokenType,
			EnrollmentID: ri.EnrollmentID,
		}
	} else {
		if req.EnrollmentID == "" || req.User == "" || req.TokenSerial == "" {
			c.RespInvalidReq(w, "no 'renew_token' or 'enrollment_id' && 'user' && 'token_serial' provided in renew request")
			return
		}

		renewInfo = &RenewInfo{
			User:         req.User,
			TokenSerial:  req.TokenSerial,
			TokenType:    req.TokenType,
			EnrollmentID: req.EnrollmentID,
		}
	}

	tokenID, err := skotty.SerialToTokenID(renewInfo.TokenType, renewInfo.TokenSerial)
	if err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("can't generate token id: %v", err))
		return
	}

	tokenState, err := c.DB.LookupUserTokenState(r.Context(), renewInfo.User, tokenID, renewInfo.EnrollmentID)
	if err != nil {
		c.RespInternalError(w, fmt.Sprintf("failed to get token state: %v", err))
		return
	}

	if tokenState != models.TokenStateActive {
		c.RespInvalidReq(w, fmt.Sprintf("invalid token state: %s", tokenState))
		return
	}

	rawAuthID, err := uuid.NewV4()
	if err != nil {
		c.RespInternalError(w, fmt.Sprintf("can't generate auth id: %v", err))
		return
	}

	authID := rawAuthID.String()
	hostToken, userToken, err := c.Auth.AuthRenewTokens(webauth.AuthInfo{
		AuthID:       authID,
		User:         renewInfo.User,
		TokenSerial:  renewInfo.TokenSerial,
		TokenType:    req.TokenType,
		TokenName:    req.TokenName,
		Hostname:     req.Hostname,
		EnrollmentID: renewInfo.EnrollmentID,
	})
	if err != nil {
		c.RespInternalError(w, fmt.Sprintf("can't generate auth token: %v", err))
		return
	}

	authURL := fmt.Sprintf(
		"https://%s/authorize/%s/#token=atv2-%s",
		r.Host, authID, base64.RawURLEncoding.EncodeToString([]byte(userToken)),
	)

	c.RespOK(w, skotty.RequestRenewRsp{
		EnrollmentID: renewInfo.EnrollmentID,
		AuthToken:    hostToken,
		AuthID:       authID,
		AuthURL:      authURL,
	})
}

func (c *Controller) enrollIssue(w http.ResponseWriter, r *http.Request) {
	var req skotty.IssueEnrollmentReq
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("failed to parse request: %v", err))
		return
	}

	authInfo, err := c.Auth.ParseAuthToken(req.AuthToken, models.AuthKindEnrollHost)
	if err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("invalid auth token: %v", err))
		return
	}

	issueReq := CertsReq{
		TokenType:       req.TokenType,
		AttestationCert: req.AttestationCert,
		Certificates:    req.Certificates,
	}
	issueInfo, err := c.issueCerts(r.Context(), authInfo, issueReq)
	if err != nil {
		if errors.Is(err, db.ErrNotFound) {
			err := &skotty.ServiceError{
				Code: skotty.ServiceErrorUnauthorizedRequest,
				Msg:  "enrollment not authorized",
			}
			c.RespError(w, err, http.StatusBadRequest)
			return
		}

		c.RespInternalError(w, fmt.Sprintf("failed to issue certificates: %v", err))
		return
	}

	var staffErr string
	if err := c.addStaffKeys(r.Context(), issueInfo); err != nil {
		staffErr = err.Error()
	}

	renewToken, err := c.Auth.RenewToken(webauth.RenewInfo{
		AuthID:       issueInfo.AuthID,
		User:         issueInfo.User,
		TokenSerial:  issueInfo.TokenSerial,
		TokenType:    issueInfo.TokenType,
		EnrollmentID: issueInfo.EnrollmentID,
	})
	if err != nil {
		c.Log.Error("failed to generate renewal token", log.Any("auth_info", authInfo))
		c.RespInternalError(w, fmt.Sprintf("failed to issue enrollment: %v", err))
		return
	}

	if err := c.DB.DropAuthorization(r.Context(), issueInfo.AuthID); err != nil {
		c.Log.Error("failed to drop authorization", log.Any("auth_info", authInfo))
		c.RespInternalError(w, fmt.Sprintf("failed to issue enrollment: %v", err))
		return
	}

	revokedTokens, err := c.DB.ScheduleRevokeTokenID(r.Context(), issueInfo.TokenID)
	if err != nil {
		c.Log.Error("failed to revoke token", log.Any("auth_info", authInfo), log.Error(err))

		// TODO(buglloc): use different err codes
		c.RespInternalError(w, fmt.Sprintf("failed to revoke previous tokens: %v", err))
		return
	}

	revokeMsg := fmt.Sprintf("revoking in favor to: %s/%s/%s", issueInfo.User, issueInfo.TokenID, issueInfo.EnrollmentID)
	for _, token := range revokedTokens {
		c.AuditLog.Log(token.ID, token.EnrollID, revokeMsg)
	}

	token := models.Token{
		ID:         issueInfo.TokenID,
		User:       issueInfo.User,
		Name:       issueInfo.TokenName,
		EnrollID:   issueInfo.EnrollmentID,
		TokenType:  issueInfo.TokenType,
		TokenState: models.TokenStateActive,
		ExpiresAt:  issueInfo.ExpiresAt,
	}

	err = c.DB.InsertToken(r.Context(), token)
	if err != nil {
		c.Log.Error("failed to insert token", log.Any("auth_info", authInfo))

		// TODO(buglloc): use different err codes
		c.RespInternalError(w, fmt.Sprintf("failed to save new token: %v", err))
		return
	}

	c.Mailer.TokenEnroll(r.Context(), token)
	c.AuditLog.Log(issueInfo.TokenID, issueInfo.EnrollmentID, fmt.Sprintf("token issued by auth id: %s", issueInfo.AuthID))
	c.RespOK(w, skotty.IssueEnrollmentRsp{
		RenewToken:    renewToken,
		ExpiresAt:     issueInfo.ExpiresAt,
		Certificates:  issueInfo.Certificates,
		KeysUpdated:   staffErr == "",
		KeysUpdateErr: staffErr,
		EnrollInfo: skotty.EnrollmentInfo{
			TokenSerial: issueInfo.TokenSerial,
			EnrollID:    issueInfo.EnrollmentID,
			User:        issueInfo.User,
		},
	})
}

func (c *Controller) addStaffKeys(ctx context.Context, issueInfo *CertsRsp) error {
	staffKeys, err := c.StaffAPI.UserKeys(ctx, issueInfo.User)
	if err != nil {
		return fmt.Errorf("unable to get staff user keys: %w", err)
	}

	existedKeys := make(map[string]struct{}, len(staffKeys))
	for _, key := range staffKeys {
		existedKeys[key.Fingerprint] = struct{}{}
	}

	var toAdd []staff.SSHKey
	for _, cert := range issueInfo.Certificates {
		switch cert.Type {
		case skotty.CertTypeLegacy, skotty.CertTypeInsecure, skotty.CertTypeSecure:
		default:
			continue
		}

		if _, ok := existedKeys[cert.AuthorizedKey.Fingerprint]; ok {
			continue
		}

		toAdd = append(toAdd, staff.SSHKey{
			Key:         cert.AuthorizedKey.Key,
			Description: cert.AuthorizedKey.Description,
		})
	}

	c.Log.Info("add keys to the staff",
		log.String("user", issueInfo.User),
		log.String("enrollment_id", issueInfo.EnrollmentID),
		log.Any("keys", toAdd))
	err = c.Staff.AddSSHKey(ctx, issueInfo.UserTicket, toAdd...)
	if err != nil {
		c.Log.Error("unable to add keys to the staff", log.String("user", issueInfo.User),
			log.String("enrollment_id", issueInfo.EnrollmentID),
			log.Any("keys", toAdd),
			log.Error(err))
	}
	return err
}

func (c *Controller) approveRenewIssue(w http.ResponseWriter, r *http.Request) {
	var req skotty.ApproveRenewReq
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("failed to parse request: %v", err))
		return
	}

	authInfo, err := c.Auth.ParseAuthToken(req.AuthToken, models.AuthKindRenewHost)
	if err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("invalid auth token: %v", err))
		return
	}

	if authInfo.EnrollmentID != chi.URLParam(r, "enrollID") {
		c.RespInvalidReq(w, fmt.Sprintf("invalid enroll id: %s != %s", authInfo.EnrollmentID, chi.URLParam(r, "enrollID")))
		return
	}

	signCert, err := certutil.PemToCert(req.SignKey)
	if err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("failed to parse token certificate: %v", err))
		return
	}

	certInfo, err := c.CA.Verify(skotty.CertTypeRenew, signCert, []byte(authInfo.AuthID), req.Sign)
	if err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("invalid certificate: %v", err))
		return
	}

	if certInfo.User == "" {
		c.RespInvalidReq(w, "invalid certificate: no user in certificate")
		return
	}

	tokenID, err := skotty.SerialToTokenID(authInfo.TokenType, authInfo.TokenSerial)
	if err != nil {
		c.RespInvalidReq(w, "invalid certificate: can't generate token id")
		return
	}

	if tokenID != certInfo.TokenID {
		c.RespInvalidReq(w, "invalid certificate: invalid token id")
		return
	}

	if authInfo.EnrollmentID != certInfo.EnrollID {
		c.RespInvalidReq(w, "invalid certificate: invalid enroll id")
		return
	}

	_, err = c.DB.LookupAuthorization(r.Context(), authInfo.AuthID)
	switch err {
	case db.ErrNotFound:
	case nil:
		c.RespInvalidReq(w, "already authorized")
		return
	default:
		c.RespInternalError(w, fmt.Sprintf("can't check auth state: %v", err))
		return
	}

	authorization := &models.Authorization{
		ID:   authInfo.AuthID,
		User: certInfo.User,
	}
	authorization.Sign, err = c.Auth.SignAuthorization(authorization)
	if err != nil {
		c.Log.Error("authorization sign failed",
			log.String("auth_id", authInfo.AuthID),
			log.String("user", certInfo.User),
			log.Error(err))
		c.RespInternalError(w, fmt.Sprintf("failed to sign your authorization approve: %v", err))
		return
	}

	err = c.DB.IssueAuthorization(r.Context(), authorization)
	if err != nil {
		c.RespInternalError(w, fmt.Sprintf("failed to approve enrollment request: %v", err))
		return
	}

	w.Header().Set("Content-Type", "application/json")
	_, _ = w.Write([]byte(`{}`))
}

func (c *Controller) renewIssue(w http.ResponseWriter, r *http.Request) {
	var req skotty.IssueRenewReq
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("failed to parse request: %v", err))
		return
	}

	authInfo, err := c.Auth.ParseAuthToken(req.AuthToken, models.AuthKindRenewHost)
	if err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("invalid auth token: %v", err))
		return
	}

	issueReq := CertsReq{
		TokenType:       req.TokenType,
		AttestationCert: req.AttestationCert,
		Certificates:    req.Certificates,
	}
	issueInfo, err := c.issueCerts(r.Context(), authInfo, issueReq)
	if err != nil {
		if errors.Is(err, db.ErrNotFound) {
			err := &skotty.ServiceError{
				Code: skotty.ServiceErrorUnauthorizedRequest,
				Msg:  "enrollment not authorized",
			}
			c.RespError(w, err, http.StatusBadRequest)
			return
		}

		c.RespInternalError(w, fmt.Sprintf("failed to issue certificates: %v", err))
		return
	}

	renewToken, err := c.Auth.RenewToken(webauth.RenewInfo{
		AuthID:       issueInfo.AuthID,
		User:         issueInfo.User,
		TokenSerial:  issueInfo.TokenSerial,
		TokenType:    issueInfo.TokenType,
		EnrollmentID: issueInfo.EnrollmentID,
	})
	if err != nil {
		c.Log.Error("failed to generate renewal token", log.Any("auth_info", authInfo))
		c.RespInternalError(w, fmt.Sprintf("failed to issue renewal: %v", err))
		return
	}

	if err := c.DB.DropAuthorization(r.Context(), issueInfo.AuthID); err != nil {
		c.Log.Error("failed to drop authorization", log.Any("auth_info", authInfo))
		c.RespInternalError(w, fmt.Sprintf("failed to issue renewal: %v", err))
		return
	}

	certToRevoke := make(map[skotty.CertType]string, len(issueInfo.Certificates))
	for _, cert := range issueInfo.Certificates {
		certToRevoke[cert.Type] = cert.Serial
	}

	tokenFullID := models.TFID(issueInfo.User, issueInfo.TokenID, issueInfo.EnrollmentID)
	revokedCerts, err := c.DB.SetCertsState(r.Context(), tokenFullID, certToRevoke, models.CertStateRenewed)
	if err != nil {
		c.Log.Error("failed to revoke certs", log.Any("auth_info", authInfo))

		// TODO(buglloc): use different err codes
		c.RespInternalError(w, fmt.Sprintf("failed to revoke previous certs: %v", err))
		return
	}

	for _, cert := range revokedCerts {
		certType := cert.CertType.String()
		revokeMsg := fmt.Sprintf("certificate %s@%s renewed in favor to: %s@%s", cert.Serial, certType, certToRevoke[cert.CertType], certType)
		c.AuditLog.Log(issueInfo.TokenID, issueInfo.EnrollmentID, revokeMsg)
	}

	token := models.Token{
		ID:        issueInfo.TokenID,
		User:      issueInfo.User,
		TokenType: issueInfo.TokenType,
		Name:      issueInfo.TokenName,
		EnrollID:  issueInfo.EnrollmentID,
		ExpiresAt: issueInfo.ExpiresAt,
	}
	err = c.DB.UpdateToken(r.Context(), token)

	if err != nil {
		c.Log.Error("failed to update token", log.Any("auth_info", authInfo))

		// TODO(buglloc): use different err codes
		c.RespInternalError(w, fmt.Sprintf("failed to update token: %v", err))
		return
	}

	c.Mailer.TokenRenew(r.Context(), token)
	c.AuditLog.Log(issueInfo.TokenID, issueInfo.EnrollmentID, fmt.Sprintf("token renewed by auth id: %s", issueInfo.AuthID))
	c.RespOK(w, skotty.IssueRenewRsp{
		RenewToken:   renewToken,
		ExpiresAt:    issueInfo.ExpiresAt,
		Certificates: issueInfo.Certificates,
		EnrollInfo: skotty.EnrollmentInfo{
			TokenSerial: issueInfo.TokenSerial,
			EnrollID:    issueInfo.EnrollmentID,
			User:        issueInfo.User,
		},
	})
}

func (c *Controller) issueCerts(ctx context.Context, authInfo webauth.AuthInfo, req CertsReq) (*CertsRsp, error) {
	if authInfo.TokenType != req.TokenType {
		return nil, fmt.Errorf("token type mismatch: %s != %s", authInfo.TokenType, req.TokenType)
	}

	if len(req.AttestationCert) == 0 {
		return nil, errors.New("no attestation certificate provided")
	}

	if len(req.Certificates) == 0 {
		return nil, errors.New("no certificates requested")
	}

	dbAuthInfo, err := c.DB.LookupAuthorization(ctx, authInfo.AuthID)
	if err != nil {
		return nil, fmt.Errorf("can't check auth state: %w", err)
	}

	if dbAuthInfo.Used {
		return nil, errors.New("auth already used")
	}

	dbAuthOk, err := c.Auth.CheckAuthorizationSign(dbAuthInfo)
	if err != nil {
		return nil, fmt.Errorf("can't check auth sign: %w", err)
	}

	if !dbAuthOk {
		return nil, errors.New("forged authorization detected, please contact security@yandex-team.ru")
	}

	if authInfo.User == "" {
		authInfo.User = dbAuthInfo.User
	}

	if authInfo.User != dbAuthInfo.User {
		return nil, fmt.Errorf("invalid user: %s != %s", authInfo.User, dbAuthInfo.User)
	}

	tokenAttestCert, err := certutil.PemToCert(req.AttestationCert)
	if err != nil {
		return nil, fmt.Errorf("failed to parse token certificate: %w", err)
	}

	tokenSigner, err := c.Signer.Signer(req.TokenType)
	if err != nil {
		return nil, err
	}

	tokenID, err := skotty.SerialToTokenID(authInfo.TokenType, authInfo.TokenSerial)
	if err != nil {
		return nil, err
	}

	enrollID := authInfo.EnrollmentID
	tokenFullID := models.TFID(authInfo.User, tokenID, enrollID)

	validAfter := time.Now()
	validBefore := validAfter.Add(tokenSigner.CertLifetime())
	tokenValidBefore := validBefore

	certs := make([]skotty.IssuedCertificate, len(req.Certificates))
	spottedCerts := make(map[skotty.CertType]struct{}, len(req.Certificates))
	for i, reqCert := range req.Certificates {
		if _, ok := spottedCerts[reqCert.Type]; ok {
			return nil, fmt.Errorf("duplicate cert request: %s", reqCert.Type)
		}

		certAttestCert, err := certutil.PemToCert(reqCert.AttestationCert)
		if err != nil {
			return nil, fmt.Errorf("failed to parse attestation cert for %s: %w", reqCert.Type, err)
		}

		spottedCerts[reqCert.Type] = struct{}{}
		attest, err := tokenSigner.AttestSlot(
			reqCert.Type,
			tokenAttestCert,
			certAttestCert,
		)
		if err != nil {
			return nil, fmt.Errorf("attestation failed cert of type %s: %w", reqCert.Type, err)
		}

		if attest.Serial != authInfo.TokenSerial {
			return nil, fmt.Errorf("attestation failed invalid serial number: %s (expected) != %s (actual)", authInfo.TokenSerial, attest.Serial)
		}

		cert, err := certutil.PemToCert(reqCert.Cert)
		if err != nil {
			return nil, fmt.Errorf("failed to parse cert of type %s: %w", reqCert.Type, err)
		}

		if !reflect.DeepEqual(cert.PublicKey, certAttestCert.PublicKey) {
			return nil, fmt.Errorf("public key from cert of type %s doesn't matched public key from slot attestation certificate", reqCert.Type)
		}

		var issuedCert *signer.SSHCert
		var authorizedKey skotty.AuthorizedKey
		//TODO(buglloc): ugly
		switch reqCert.Type {
		case skotty.CertTypeLegacy:
			sshPub, err := ssh.NewPublicKey(cert.PublicKey)
			if err != nil {
				return nil, fmt.Errorf("failed to parse legacy pub key: %w", err)
			}

			cert.SerialNumber = signer.NewSerial()
			cert.NotBefore = validAfter
			// 4.1.2.5. Validity To indicate that a certificate has no well-defined expiration date, the notAfter SHOULD be assigned the GeneralizedTime value of 99991231235959Z.
			cert.NotAfter = time.Date(9999, 12, 31, 23, 59, 59, 0, time.UTC)
			issuedCert = &signer.SSHCert{
				Cert:   cert,
				SSHPub: sshPub,
			}

			authorizedKey.Description = fmt.Sprintf("Skotty %s key on %s@%s", reqCert.Type, authInfo.TokenSerial, authInfo.Hostname)
			authorizedKey.Fingerprint = ssh.FingerprintSHA256(sshPub)
			authorizedKey.Key = fmt.Sprintf("%s # %s",
				string(bytes.TrimSpace(ssh.MarshalAuthorizedKey(sshPub))),
				authorizedKey.Description,
			)
		case skotty.CertTypeRenew:
			csr := &signer.CertificateRequest{
				PublicKey:  cert.PublicKey,
				EnrollID:   enrollID,
				TokenID:    tokenID,
				User:       authInfo.User,
				ValidAfter: validAfter,
				// added additional gap for renew certificates to minimize troubles
				ValidBefore: tokenValidBefore.Add(renewTTLGap),
			}

			issuedCert, err = tokenSigner.IssueCertificate(reqCert.Type, csr)
			if err != nil {
				return nil, fmt.Errorf("failed to issue certificate of type %s: %w", reqCert.Type, err)
			}
		default:
			csr := &signer.CertificateRequest{
				PublicKey:   cert.PublicKey,
				EnrollID:    enrollID,
				TokenID:     tokenID,
				User:        authInfo.User,
				ValidAfter:  validAfter,
				ValidBefore: validBefore,
			}

			issuedCert, err = tokenSigner.IssueCertificate(reqCert.Type, csr)
			if err != nil {
				return nil, fmt.Errorf("failed to issue certificate of type %s: %w", reqCert.Type, err)
			}

			sshCert, ok := issuedCert.SSHPub.(*ssh.Certificate)
			if !ok {
				return nil, fmt.Errorf("signer returns non-cert key for reqested type: %s", reqCert.Type)
			}

			authorizedKey.Description = fmt.Sprintf("Skotty %s CA", strings.Title(reqCert.Type.String()))
			authorizedKey.Fingerprint = ssh.FingerprintSHA256(sshCert.SignatureKey)
			authorizedKey.Key = fmt.Sprintf("cert-authority,principals=%q %s # %s",
				authInfo.User,
				string(bytes.TrimSpace(ssh.MarshalAuthorizedKey(sshCert.SignatureKey))),
				authorizedKey.Description,
			)
		}

		pemCert := bytes.TrimSpace(certutil.CertToPem(issuedCert.Cert))
		sshFp := sshutil.Fingerprint(issuedCert.SSHPub)
		sshCert := bytes.TrimSpace(ssh.MarshalAuthorizedKey(issuedCert.SSHPub))
		serial := issuedCert.Cert.SerialNumber.String()

		err = c.DB.InsertCertificate(ctx, models.Certificate{
			TokenFullID:    tokenFullID,
			TokenType:      req.TokenType,
			CertType:       reqCert.Type,
			Principal:      authInfo.User,
			Serial:         serial,
			CAFingerprint:  issuedCert.CAFingerprint,
			Cert:           pemCert,
			SSHCert:        sshCert,
			SSHFingerprint: sshFp,
			ValidAfter:     issuedCert.Cert.NotBefore.Unix(),
			ValidBefore:    issuedCert.Cert.NotAfter.Unix(),
		})
		if err != nil {
			return nil, fmt.Errorf("failed to save certificate of type %s: %w", reqCert.Type, err)
		}

		c.AuditLog.Log(tokenID, enrollID, fmt.Sprintf("certificate issued: %s@%s", serial, reqCert.Type))
		certs[i] = skotty.IssuedCertificate{
			Serial:        serial,
			AuthorizedKey: authorizedKey,
			HostID:        reqCert.HostID,
			Type:          reqCert.Type,
			Cert:          pemCert,
			SSHCert:       sshCert,
		}
	}

	return &CertsRsp{
		AuthID:       authInfo.AuthID,
		User:         dbAuthInfo.User,
		UserTicket:   dbAuthInfo.UserTicket,
		ExpiresAt:    tokenValidBefore.Unix(),
		TokenSerial:  authInfo.TokenSerial,
		TokenID:      tokenID,
		TokenType:    authInfo.TokenType,
		TokenName:    authInfo.TokenName,
		EnrollmentID: enrollID,
		Certificates: certs,
	}, nil
}

func (c *Controller) Shutdown(_ context.Context) {}

func (c *Controller) RespOK(w http.ResponseWriter, rsp interface{}) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	w.WriteHeader(http.StatusOK)
	_ = json.NewEncoder(w).Encode(rsp)
}

func (c *Controller) RespError(w http.ResponseWriter, err *skotty.ServiceError, code int) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	w.WriteHeader(code)
	_ = json.NewEncoder(w).Encode(err)
	if err.Code != skotty.ServiceErrorUnauthorizedRequest {
		c.Log.Error("failed to process request", log.Int("code", code), log.Error(err))
	}
}

func (c *Controller) RespInvalidReq(w http.ResponseWriter, msg string) {
	err := &skotty.ServiceError{
		Code: skotty.ServiceErrorInvalidRequest,
		Msg:  msg,
	}
	c.RespError(w, err, http.StatusBadRequest)
}

func (c *Controller) RespInternalError(w http.ResponseWriter, msg string) {
	err := &skotty.ServiceError{
		Code: skotty.ServiceErrorInternalError,
		Msg:  msg,
	}
	c.RespError(w, err, http.StatusInternalServerError)
}
