package roboapi

import (
	"bytes"
	"context"
	"encoding/base64"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"path"
	"time"

	"github.com/go-chi/chi/v5"
	"golang.org/x/crypto/ssh"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/skotty/libs/certutil"
	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/service/internal/app/controller"
	"a.yandex-team.ru/security/skotty/service/internal/app/env"
	"a.yandex-team.ru/security/skotty/service/internal/signer"
)

type Controller struct {
	*env.Env
}

func NewController(e *env.Env) (controller.Controller, error) {
	if e.Roles.RoboSSH == "" {
		return nil, errors.New("no robossh role provided")
	}

	return &Controller{
		Env: e,
	}, nil
}

func (c *Controller) BuildRoute(r chi.Router) {
	r.Use(authMiddleware(c.BlackBox, c.TVM))

	r.Post("/issue", c.issue)
}

func (c *Controller) issue(w http.ResponseWriter, r *http.Request) {
	var req skotty.IssueRoboCertificatesReq
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("failed to parse request: %v", err))
		return
	}

	issueInfo, err := c.issueCerts(r.Context(), req)
	if err != nil {
		c.RespInvalidReq(w, fmt.Sprintf("failed to issue certificates: %v", err))
		return
	}

	c.RespOK(w, skotty.IssueRoboCertificatesRsp{
		ExpiresAt:    issueInfo.ExpiresAt,
		Certificates: issueInfo.Certificates,
	})
}

func (c *Controller) issueCerts(ctx context.Context, req skotty.IssueRoboCertificatesReq) (*CertsRsp, error) {
	tokenSigner, err := c.Signer.Signer(skotty.TokenTypeRoboMemory)
	if err != nil {
		return nil, err
	}

	tokenID, err := skotty.SerialToTokenID(skotty.TokenTypeRoboMemory, req.TokenSerial)
	if err != nil {
		return nil, err
	}

	user, ok := userFromContext(ctx)
	if !ok {
		return nil, errors.New("can't get user from context")
	}

	enrollID := base64.RawURLEncoding.EncodeToString([]byte(user.IP))
	validAfter := time.Now()
	validBefore := validAfter.Add(tokenSigner.CertLifetime())
	tokenValidBefore := validBefore

	certs := make([]skotty.IssuedCertificate, len(req.Certificates))
	spottedCerts := make(map[skotty.CertType]struct{}, len(req.Certificates))
	for i, reqCert := range req.Certificates {
		if _, ok := spottedCerts[reqCert.Type]; ok {
			return nil, fmt.Errorf("duplicate cert request: %s", reqCert.Type)
		}
		spottedCerts[reqCert.Type] = struct{}{}

		haveAccess, err := c.CheckCaRole(user, reqCert.Type)
		if err != nil {
			return nil, err
		}

		if !haveAccess {
			return nil, fmt.Errorf("user %q requested CA %s, but doesn't have IDM role: %s", user.Login, reqCert.Type, c.certTypeToCaRole(reqCert.Type))
		}

		cert, err := certutil.PemToCert(reqCert.Cert)
		if err != nil {
			return nil, fmt.Errorf("failed to parse cert of type %s: %w", reqCert.Type, err)
		}

		switch reqCert.Type {
		case skotty.CertTypeInsecure, skotty.CertTypeSecure, skotty.CertTypeSudo:
			// ok
		default:
			return nil, fmt.Errorf("requested unsupported cert type: %s", reqCert.Type)
		}

		csr := &signer.CertificateRequest{
			PublicKey:   cert.PublicKey,
			EnrollID:    enrollID,
			TokenID:     tokenID,
			User:        user.Login,
			ValidAfter:  validAfter,
			ValidBefore: validBefore,
		}

		issuedCert, err := tokenSigner.IssueCertificate(reqCert.Type, csr)
		if err != nil {
			return nil, fmt.Errorf("failed to issue certificate of type %s: %w", reqCert.Type, err)
		}

		certs[i] = skotty.IssuedCertificate{
			Serial:  issuedCert.Cert.SerialNumber.String(),
			HostID:  reqCert.HostID,
			Type:    reqCert.Type,
			Cert:    bytes.TrimSpace(certutil.CertToPem(issuedCert.Cert)),
			SSHCert: bytes.TrimSpace(ssh.MarshalAuthorizedKey(issuedCert.SSHPub)),
		}
	}

	return &CertsRsp{
		User:         user.Login,
		ExpiresAt:    tokenValidBefore.Unix(),
		TokenSerial:  req.TokenSerial,
		TokenID:      tokenID,
		TokenType:    skotty.TokenTypeRoboMemory,
		TokenName:    req.TokenName,
		EnrollmentID: enrollID,
		Certificates: certs,
	}, nil
}

func (c *Controller) Shutdown(_ context.Context) {}

func (c *Controller) CheckCaRole(user User, certType skotty.CertType) (bool, error) {
	ok, err := user.Roles.CheckUserRole(user.UserTicket, c.certTypeToCaRole(certType), nil)
	if err != nil {
		return false, fmt.Errorf("unable to check check user roles: %w", err)
	}

	return ok, nil
}

func (c *Controller) certTypeToCaRole(certType skotty.CertType) string {
	return path.Join(c.Roles.RoboSSH, "ca_access", certType.String()) + "/"
}

func (c *Controller) RespOK(w http.ResponseWriter, rsp interface{}) {
	respOK(w, rsp)
}

func (c *Controller) RespError(w http.ResponseWriter, code int, err *skotty.ServiceError) {
	respError(w, code, err)
	switch err.Code {
	case skotty.ServiceErrorUnauthorizedRequest:
	case skotty.ServiceErrorInvalidRequest:
		c.Log.Warn("failed to process request", log.Int("code", code), log.Error(err))
	default:
		c.Log.Error("failed to process request", log.Int("code", code), log.Error(err))
	}
}

func (c *Controller) RespInvalidReq(w http.ResponseWriter, msg string) {
	err := &skotty.ServiceError{
		Code: skotty.ServiceErrorInvalidRequest,
		Msg:  msg,
	}
	c.RespError(w, http.StatusBadRequest, err)
}

func (c *Controller) RespInternalError(w http.ResponseWriter, msg string) {
	err := &skotty.ServiceError{
		Code: skotty.ServiceErrorInternalError,
		Msg:  msg,
	}
	c.RespError(w, http.StatusInternalServerError, err)
}

func respOK(w http.ResponseWriter, rsp interface{}) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	w.WriteHeader(http.StatusOK)
	_ = json.NewEncoder(w).Encode(rsp)
}

func respError(w http.ResponseWriter, code int, err *skotty.ServiceError) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	w.WriteHeader(code)
	_ = json.NewEncoder(w).Encode(err)
}
