package roboapi

import (
	"context"
	"fmt"
	"net/http"
	"strings"

	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/service/internal/oauth"
)

type contextKey string

const contextUserKey = contextKey("user")

type User struct {
	UID        blackbox.ID
	Login      string
	IP         string
	UserTicket *tvm.CheckedUserTicket
	Roles      *tvm.Roles
}

func authMiddleware(bbClient blackbox.Client, tvmClient tvm.Client) func(http.Handler) http.Handler {
	return func(next http.Handler) http.Handler {
		fn := func(w http.ResponseWriter, r *http.Request) {
			authHeader := r.Header.Get("Authorization")
			if authHeader == "" {
				respError(w, http.StatusUnauthorized, &skotty.ServiceError{
					Code: skotty.ServiceErrorUnauthorizedRequest,
					Msg:  "no Authorization header",
				})
				return
			}

			if !strings.HasPrefix(authHeader, "OAuth ") {
				respError(w, http.StatusUnauthorized, &skotty.ServiceError{
					Code: skotty.ServiceErrorUnauthorizedRequest,
					Msg:  "invalid Authorization header",
				})
				return
			}

			rsp, err := bbClient.OAuth(r.Context(), blackbox.OAuthRequest{
				OAuthToken:    authHeader[6:],
				UserIP:        r.RemoteAddr,
				GetUserTicket: true,
				Scopes:        oauth.RequiredScopes,
			})

			if err != nil {
				respError(w, http.StatusUnauthorized, &skotty.ServiceError{
					Code: skotty.ServiceErrorUnauthorizedRequest,
					Msg:  fmt.Sprintf("auth failed: %v", err),
				})
				return
			}

			if rsp.ClientID != oauth.ClientID {
				respError(w, http.StatusForbidden, &skotty.ServiceError{
					Code: skotty.ServiceErrorUnauthorizedRequest,
					Msg:  "invalid client id",
				})
				return
			}

			if !strings.HasPrefix(rsp.User.Login, "robot-") && !strings.HasPrefix(rsp.User.Login, "zomb-") {
				respError(w, http.StatusForbidden, &skotty.ServiceError{
					Code: skotty.ServiceErrorUnauthorizedRequest,
					Msg:  "only zomb- or robot- users allowed",
				})
				return
			}

			ticket, err := tvmClient.CheckUserTicket(r.Context(), rsp.UserTicket)
			if err != nil {
				respError(w, http.StatusForbidden, &skotty.ServiceError{
					Code: skotty.ServiceErrorUnauthorizedRequest,
					Msg:  fmt.Sprintf("blackbox return invalid user ticket: %v", err),
				})
				return
			}

			roles, err := tvmClient.GetRoles(r.Context())
			if err != nil {
				respError(w, http.StatusForbidden, &skotty.ServiceError{
					Code: skotty.ServiceErrorUnauthorizedRequest,
					Msg:  fmt.Sprintf("can't get service roles: %v", err),
				})
				return
			}

			r = r.WithContext(
				context.WithValue(
					r.Context(),
					contextUserKey,
					User{
						UID:        rsp.User.UID.ID,
						Login:      rsp.User.Login,
						IP:         r.RemoteAddr,
						UserTicket: ticket,
						Roles:      roles,
					},
				),
			)
			next.ServeHTTP(w, r)
		}
		return http.HandlerFunc(fn)
	}
}

func userFromContext(ctx context.Context) (User, bool) {
	u, ok := ctx.Value(contextUserKey).(User)
	return u, ok
}
