package env

import (
	"context"
	"fmt"
	"os"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/security/skotty/service/internal/auditlog"
	"a.yandex-team.ru/security/skotty/service/internal/config"
	"a.yandex-team.ru/security/skotty/service/internal/db"
	"a.yandex-team.ru/security/skotty/service/internal/mailer"
	"a.yandex-team.ru/security/skotty/service/internal/reminder"
	"a.yandex-team.ru/security/skotty/service/internal/revoker"
	"a.yandex-team.ru/security/skotty/service/internal/signer"
	"a.yandex-team.ru/security/skotty/service/internal/signer/anysign"
	"a.yandex-team.ru/security/skotty/service/internal/staff"
	"a.yandex-team.ru/security/skotty/service/internal/staffapi"
	"a.yandex-team.ru/security/skotty/service/internal/storage"
	"a.yandex-team.ru/security/skotty/service/internal/webauth"
	"a.yandex-team.ru/security/skotty/service/internal/yasms"
)

type Env struct {
	cfg        *config.Config
	TVM        tvm.Client
	BlackBox   blackbox.Client
	Auth       *webauth.WebAuth
	DB         *db.DB
	Mailer     *mailer.Mailer
	Storage    *storage.Storage
	Revoker    revoker.Revoker
	AuditLog   *auditlog.AuditLog
	CA         *signer.CAStorage
	Signer     *anysign.AnySigner
	Staff      *staff.Client
	StaffAPI   *staffapi.Client
	YaSMS      yasms.Client
	Reminder   reminder.Reminder
	SignSecret string
	Debug      bool
	Dev        bool
	Log        log.Logger
	Roles      config.Roles
}

func NewEnv(cfg *config.Config) *Env {
	logLvl := log.InfoLevel
	if cfg.Dev {
		logLvl = log.DebugLevel
	}

	zlog, err := zap.NewDeployLogger(logLvl)
	if err != nil {
		panic(fmt.Sprintf("failed to create logger: %s", err))
	}

	return &Env{
		cfg:   cfg,
		Debug: cfg.Dev,
		Dev:   cfg.Dev,
		Roles: cfg.Roles,
		Log:   zlog,
	}
}

func (e *Env) Initialize() error {
	var err error
	e.SignSecret = e.cfg.SignSecret

	e.TVM, err = e.newTVMClient()
	if err != nil {
		return fmt.Errorf("can't create tvm cleint: %w", err)
	}

	e.CA, err = signer.NewCAStorage(e.cfg.CAs)
	if err != nil {
		return fmt.Errorf("can't create ca storage: %w", err)
	}

	e.Signer, err = anysign.NewSigner(e.CA)
	if err != nil {
		return fmt.Errorf("can't create signer: %w", err)
	}

	e.DB, err = db.New(context.Background(), e.TVM, e.cfg.YDB)
	if err != nil {
		return fmt.Errorf("can't create database: %w", err)
	}

	e.Auth, err = webauth.NewWebAuth(e.cfg.SignSecret)
	if err != nil {
		return fmt.Errorf("can't create webauth: %w", err)
	}

	e.AuditLog = auditlog.NewAuditLog(e.DB)
	e.Mailer = mailer.NewMailer(e.cfg.Mailer.Upstream, e.cfg.Mailer.From, mailer.WithLogger(e.Log))
	e.Storage, err = storage.NewStorage(
		e.TVM,
		storage.WithEndpoint(e.cfg.S3.Endpoint),
		storage.WithBucket(e.cfg.S3.Bucket),
		storage.WithAccessKeyID(e.cfg.S3.AccessKeyID),
		storage.WithTVMClientID(e.cfg.S3.TvmID),
	)
	if err != nil {
		return fmt.Errorf("can't create storage: %w", err)
	}

	e.Staff, err = staff.NewClient(e.TVM,
		staff.WithUpstream(e.cfg.Staff.Upstream),
		staff.WithTVMID(e.cfg.Staff.ClientID))
	if err != nil {
		return fmt.Errorf("can't create staff client: %w", err)
	}

	e.StaffAPI, err = staffapi.NewClient(
		staffapi.WithUpstream(e.cfg.StaffAPI.Upstream),
		staffapi.WithAuthToken(e.cfg.StaffAPI.Token),
	)
	if err != nil {
		return fmt.Errorf("can't create staff api client: %w", err)
	}

	e.BlackBox, err = httpbb.NewIntranet(
		httpbb.WithTVM(e.TVM),
		httpbb.WithLogger(e.Log.Structured()),
	)
	if err != nil {
		return fmt.Errorf("can't create blackbox client: %w", err)
	}

	if e.cfg.Dev {
		e.YaSMS = yasms.NewDebugClient()
	} else {
		e.YaSMS, err = yasms.NewHTTPClient(
			yasms.WithTVMClient(e.TVM),
			yasms.WithEnv(yasms.Env(e.cfg.YaSMS)),
		)
		if err != nil {
			return fmt.Errorf("can't create yasms client: %w", err)
		}
	}

	if e.cfg.Revoker {
		e.Revoker, err = revoker.NewRevoker(
			revoker.WithLogger(e.Log),
			revoker.WithDB(e.DB),
			revoker.WithS3Store(e.Storage),
			revoker.WithYTLock(e.cfg.YT.Proxy, e.cfg.YT.Path, e.cfg.YT.Token),
			revoker.WithCAStore(e.CA),
			revoker.WithAuditLog(e.AuditLog),
			revoker.WithMailer(e.Mailer),
			revoker.WithStaffClient(e.Staff),
		)
		if err != nil {
			return fmt.Errorf("can't create revoker: %w", err)
		}
	} else {
		e.Log.Info("revoked disabled in config")
		e.Revoker = &revoker.NopRevoker{}
	}

	if e.cfg.Reminder {
		e.Reminder, err = reminder.NewMailReminder(
			reminder.WithLogger(e.Log),
			reminder.WithDB(e.DB),
			reminder.WithMailer(e.Mailer),
			reminder.WithYTLock(e.cfg.YT.Proxy, e.cfg.YT.Path, e.cfg.YT.Token),
		)
		if err != nil {
			return fmt.Errorf("can't create reminder: %w", err)
		}
	} else {
		e.Log.Info("reminder disabled in config")
		e.Reminder = &reminder.NopReminder{}
	}

	return nil
}

func (e *Env) Shutdown(ctx context.Context) error {
	e.Revoker.Shutdown(ctx)
	e.AuditLog.Close(ctx)
	e.Mailer.Shutdown(ctx)
	e.Reminder.Shutdown(ctx)
	return e.DB.Close(ctx)
}

func (e *Env) newTVMClient() (tvm.Client, error) {
	if e.cfg.TVM.CacheDir != "" {
		if err := os.MkdirAll(e.cfg.TVM.CacheDir, 0o700); err != nil {
			return nil, fmt.Errorf("unable to create tmm dir: %w", err)
		}
	}

	tvmSettings := tvmauth.TvmAPISettings{
		SelfID:                     e.cfg.TVM.ClientID,
		ServiceTicketOptions:       tvmauth.NewAliasesOptions(e.cfg.TVM.ClientSecret, e.cfg.TVM.Destinations),
		DiskCacheDir:               e.cfg.TVM.CacheDir,
		BlackboxEnv:                &e.cfg.TVM.Env,
		FetchRolesForIdmSystemSlug: e.cfg.TVM.IDMSlug,
		DisableDefaultUIDCheck:     true,
	}

	return tvmauth.NewAPIClient(tvmSettings, e.Log)
}
