package auditlog

import (
	"context"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/skotty/service/internal/db"
	"a.yandex-team.ru/security/skotty/service/internal/logger"
	"a.yandex-team.ru/security/skotty/service/internal/models"
)

const (
	batchSize = 100
	batchTick = 1 * time.Second
)

type AuditLog struct {
	db        *db.DB
	log       log.Logger
	ctx       context.Context
	cancelCtx context.CancelFunc
	queue     chan models.AuditMsg
	closed    chan struct{}
}

func NewAuditLog(db *db.DB) *AuditLog {
	ctx, cancelCtx := context.WithCancel(context.Background())
	out := &AuditLog{
		db:        db,
		log:       logger.Named("audit_log"),
		ctx:       ctx,
		cancelCtx: cancelCtx,
		queue:     make(chan models.AuditMsg, batchSize*3),
		closed:    make(chan struct{}),
	}

	go out.loop()
	return out
}

func (a *AuditLog) loop() {
	defer close(a.closed)

	var msgs []models.AuditMsg
	save := func(force bool) error {
		if len(msgs) == 0 {
			return nil
		}

		if len(msgs) < batchSize && !force {
			return nil
		}

		if err := a.db.InsertAuditMsgs(context.Background(), msgs...); err != nil {
			return err
		}

		msgs = msgs[:0]
		return nil
	}

	t := time.NewTicker(batchTick)
	defer t.Stop()

	for {
		forceSave := false
		forceExit := false
		select {
		case <-a.ctx.Done():
			forceExit = true
			forceSave = true
		case <-t.C:
			forceSave = true
		case e := <-a.queue:
			msgs = append(msgs, e)
		}

		if err := save(forceSave); err != nil {
			a.log.Error("failed to save audit log", log.Error(err))
		}

		if forceExit {
			return
		}
	}
}

func (a *AuditLog) Log(tokenID, enrollID, message string) {
	a.log.Info(message, log.String("token_id", tokenID), log.String("enroll_id", enrollID))

	a.queue <- models.AuditMsg{
		TS:       time.Now().UnixNano(),
		TokenID:  tokenID,
		EnrollID: enrollID,
		Message:  message,
	}
}

func (a *AuditLog) Close(ctx context.Context) {
	a.cancelCtx()

	select {
	case <-a.closed:
	case <-ctx.Done():
	}
}
