package commands

import (
	"context"
	"errors"
	"fmt"
	"os"
	"os/signal"
	"syscall"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/skotty/service/internal/app"
	"a.yandex-team.ru/security/skotty/service/internal/app/env"
	"a.yandex-team.ru/security/skotty/service/internal/config"
	"a.yandex-team.ru/security/skotty/service/internal/logger"
)

var apiArgs struct {
	Configs []string
}

var apiCmd = &cobra.Command{
	Use:          "api",
	SilenceUsage: true,
	Short:        "api commands",
	PersistentPreRunE: func(_ *cobra.Command, _ []string) error {
		if len(apiArgs.Configs) == 0 {
			return errors.New("--config is empty")
		}

		return nil
	},
}

var startCmd = &cobra.Command{
	Use:          "start",
	SilenceUsage: true,
	Short:        "Start API srv",
	RunE: func(cmd *cobra.Command, args []string) error {
		cfg, err := config.Load(apiArgs.Configs...)
		if err != nil {
			return fmt.Errorf("failed to parse config: %w", err)
		}

		appEnv := env.NewEnv(cfg)
		if err := appEnv.Initialize(); err != nil {
			return fmt.Errorf("failed to initialize app env: %w", err)
		}

		instance, err := app.NewApp(appEnv)
		if err != nil {
			return fmt.Errorf("failed to create application: %w", err)
		}

		errChan := make(chan error, 1)
		okChan := make(chan struct{}, 1)
		go func() {
			if err := instance.ListenAndServe(cfg.Addr); err != nil {
				errChan <- fmt.Errorf("failed to start application: %w", err)
			} else {
				okChan <- struct{}{}
			}
		}()

		sigChan := make(chan os.Signal, 1)
		signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM)

		defer logger.Info("service stopped")

		shutdown := func(ctx context.Context) error {
			err := instance.Shutdown(ctx)
			_ = appEnv.Shutdown(ctx)
			return err
		}

		select {
		case <-sigChan:
			logger.Info("shutting down gracefully by signal")

			ctx, cancel := context.WithTimeout(context.Background(), config.ShutdownDeadline)
			defer cancel()

			return shutdown(ctx)
		case <-okChan:
			return nil
		case err := <-errChan:
			return err
		}
	},
}

func init() {
	flags := apiCmd.PersistentFlags()
	flags.StringSliceVarP(&apiArgs.Configs, "config", "c", nil, "config path")

	apiCmd.AddCommand(startCmd)
}
