package commands

import (
	"bytes"
	"crypto/ecdsa"
	"crypto/elliptic"
	"crypto/rand"
	"crypto/x509"
	"crypto/x509/pkix"
	"encoding/json"
	"encoding/pem"
	"fmt"
	"math/big"
	"os"
	"path/filepath"
	"time"

	"github.com/spf13/cobra"
	"golang.org/x/crypto/ssh"

	"a.yandex-team.ru/security/skotty/service/internal/signer"
)

var caCmd = &cobra.Command{
	Use:   "ca",
	Short: "helpers to work with ca",
}

var caGenCmdArgs = struct {
	Name string
}{
	Name: "ca",
}

var caGenCmd = &cobra.Command{
	Use:   "gen",
	Short: "gen ca [flags] /path/to/dir",
	RunE: func(_ *cobra.Command, args []string) error {
		targetDir := "."
		if len(args) > 0 {
			targetDir = args[0]
		}
		targetDir, err := filepath.Abs(targetDir)
		if err != nil {
			return fmt.Errorf("failed normalize output dir: %w", err)
		}

		now := time.Now()
		out := signer.CertificateInfo{
			Name:      fmt.Sprintf("%s_%s", caGenCmdArgs.Name, now.Format("20060102")),
			CreatedAt: now,
		}

		ca := &x509.Certificate{
			SerialNumber: big.NewInt(now.Unix()),
			Subject: pkix.Name{
				CommonName:         out.Name,
				Country:            []string{"RU"},
				Province:           []string{"Moscow"},
				Locality:           []string{"Moscow"},
				Organization:       []string{"Yandex"},
				OrganizationalUnit: []string{"Infra"},
			},
			NotBefore:             time.Now(),
			NotAfter:              time.Now().AddDate(3, 0, 0),
			IsCA:                  true,
			ExtKeyUsage:           []x509.ExtKeyUsage{x509.ExtKeyUsageClientAuth},
			KeyUsage:              x509.KeyUsageDigitalSignature | x509.KeyUsageCertSign,
			BasicConstraintsValid: true,
		}

		caPrivKey, err := ecdsa.GenerateKey(elliptic.P256(), rand.Reader)
		if err != nil {
			return err
		}

		caPrivKeyBytes, err := x509.MarshalECPrivateKey(caPrivKey)
		if err != nil {
			return err
		}

		caBytes, err := x509.CreateCertificate(rand.Reader, ca, ca, &caPrivKey.PublicKey, caPrivKey)
		if err != nil {
			return err
		}

		caCert, err := x509.ParseCertificate(caBytes)
		if err != nil {
			return err
		}

		sshCaPub, err := ssh.NewPublicKey(caCert.PublicKey)
		if err != nil {
			return err
		}

		var buf bytes.Buffer
		// x509 public
		err = pem.Encode(&buf, &pem.Block{
			Type:  "CERTIFICATE",
			Bytes: caBytes,
		})
		if err != nil {
			return err
		}
		out.Public = make([]byte, buf.Len())
		copy(out.Public, buf.Bytes())
		buf.Reset()

		// SSH public
		buf.Write(bytes.TrimSpace(ssh.MarshalAuthorizedKey(sshCaPub)))
		buf.WriteString(" " + out.Name)
		out.SSHPublic = make([]byte, buf.Len())
		copy(out.SSHPublic, buf.Bytes())
		buf.Reset()

		outBytes, err := json.Marshal(out)
		if err != nil {
			return fmt.Errorf("failed to marshal CA info: %w", err)
		}

		targetPath := filepath.Join(targetDir, out.Name+"_pubs.json")
		err = os.WriteFile(targetPath, outBytes, 0o600)
		if err != nil {
			return fmt.Errorf("failed to save file: %w", err)
		}
		fmt.Printf("new CA (pubs) saved into: %s\n", targetPath)

		// private
		err = pem.Encode(&buf, &pem.Block{
			Type:  "EC PRIVATE KEY",
			Bytes: caPrivKeyBytes,
		})
		if err != nil {
			return err
		}
		out.Private = make([]byte, buf.Len())
		copy(out.Private, buf.Bytes())
		buf.Reset()

		outBytes, err = json.Marshal(out)
		if err != nil {
			return fmt.Errorf("failed to marshal CA info: %w", err)
		}

		targetPath = filepath.Join(targetDir, out.Name+".json")
		err = os.WriteFile(targetPath, outBytes, 0o600)
		if err != nil {
			return fmt.Errorf("failed to save file: %w", err)
		}

		fmt.Printf("new CA (full) saved into: %s\n", targetPath)
		return nil
	},
}

func init() {
	flags := caGenCmd.PersistentFlags()
	flags.StringVar(&caGenCmdArgs.Name, "name", caGenCmdArgs.Name, "CA name")

	caCmd.AddCommand(
		caGenCmd,
	)
}
