package config

import (
	"context"
	"errors"
	"fmt"
	"os"
	"time"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"

	"a.yandex-team.ru/library/go/yandex/awstvm"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/skotty/libs/skotty"
)

const ShutdownDeadline = 5 * time.Minute

type CA struct {
	Type    skotty.CertType `yaml:"type"`
	Prev    string          `yaml:"prev"`
	Current string          `yaml:"current"`
	Next    string          `yaml:"next"`
}

type Mailer struct {
	Upstream string `yaml:"upstream"`
	From     string `yaml:"from"`
}

type YDB struct {
	Endpoint string `yaml:"endpoint"`
	Database string `yaml:"database"`
	Path     string `yaml:"path"`
}

type S3 struct {
	Endpoint    string       `yaml:"endpoint"`
	Bucket      string       `yaml:"bucket"`
	AccessKeyID string       `yaml:"access_key_id"`
	TvmID       tvm.ClientID `yaml:"tvm_id"`
}

type YT struct {
	Proxy string `yaml:"proxy"`
	Path  string `yaml:"path"`
	Token string `yaml:"token"`
}

type StaffAPI struct {
	Upstream string `yaml:"upstream"`
	Token    string `yaml:"token"`
}

type YaSMS struct {
	Upstream    string       `yaml:"upstream"`
	Sender      string       `yaml:"sender"`
	TVMClientID tvm.ClientID `yaml:"tvm_id"`
}

type TVM struct {
	ClientID     tvm.ClientID            `yaml:"tvm_id"`
	ClientSecret string                  `yaml:"-"`
	CacheDir     string                  `yaml:"cache_dir"`
	Destinations map[string]tvm.ClientID `yaml:"dests"`
	Env          tvm.BlackboxEnv         `yaml:"env"`
	IDMSlug      string                  `yaml:"idm_slug"`
}

type Roles struct {
	Admin   string `yaml:"admin"`
	Viewer  string `yaml:"viewer"`
	RoboSSH string `yaml:"robossh"`
}

type Staff struct {
	Upstream string       `yaml:"upstream"`
	ClientID tvm.ClientID `yaml:"client_id"`
}

type Config struct {
	Dev        bool     `yaml:"dev"`
	Revoker    bool     `yaml:"revoker"`
	Reminder   bool     `yaml:"reminder"`
	SignSecret string   `yaml:"-"`
	Addr       string   `yaml:"addr"`
	CAs        []CA     `yaml:"ca"`
	YDB        YDB      `yaml:"ydb"`
	Mailer     Mailer   `yaml:"mail"`
	S3         S3       `yaml:"s3"`
	YT         YT       `yaml:"yt"`
	StaffAPI   StaffAPI `yaml:"staff_api"`
	Staff      Staff    `yaml:"staff"`
	YaSMS      YaSMS    `yaml:"ya_sms"`
	TVM        TVM      `yaml:"tvm"`
	Roles      Roles    `yaml:"roles"`
}

func (c *Config) Validate() error {
	isFileExists := func(filepath string) error {
		if filepath == "" {
			return errors.New("empty path")
		}

		_, err := os.Stat(filepath)
		return err
	}

	certTypes := make(map[skotty.CertType]struct{})
	for _, ca := range c.CAs {
		err := func() error {
			if _, exists := certTypes[ca.Type]; exists {
				return fmt.Errorf("duplicate CAs: %s", ca.Type)
			}

			if err := isFileExists(ca.Current); err != nil {
				return fmt.Errorf("failed to check 'current' certificate: %w", err)
			}

			if err := isFileExists(ca.Next); err != nil {
				return fmt.Errorf("failed to check 'next' certificate: %w", err)
			}

			if ca.Prev == "" {
				return nil
			}

			if err := isFileExists(ca.Next); err != nil {
				return fmt.Errorf("failed to check 'prev' certificate: %w", err)
			}

			return nil
		}()

		if err != nil {
			return fmt.Errorf("CA %s validation failed: %w", ca.Type, err)
		}
	}

	return nil
}

func Load(configs ...string) (*Config, error) {
	cfg := &Config{
		Addr:       "localhost:9000",
		Revoker:    true,
		Reminder:   true,
		SignSecret: os.Getenv("SIGN_SECRET"),
		S3: S3{
			Endpoint:    "s3.mds.yandex.net",
			Bucket:      "skotty",
			AccessKeyID: "Y6UGRgZiUgtec2Kna1Kg",
			TvmID:       awstvm.S3TvmID,
		},
		YT: YT{
			Proxy: "locke",
			Path:  "//home/security/skotty/dev",
			Token: os.Getenv("YT_TOKEN"),
		},
		StaffAPI: StaffAPI{
			Token: os.Getenv("STAFF_TOKEN"),
		},
		Staff: Staff{
			Upstream: "https://staff.yandex-team.ru",
			ClientID: 2000054,
		},
		YaSMS: YaSMS{
			Sender: "skotty",
		},
		TVM: TVM{
			ClientSecret: os.Getenv("TVM_SECRET"),
		},
		Roles: Roles{
			Admin:   "/role/admin/",
			Viewer:  "/role/viewer/",
			RoboSSH: "/role/robossh/",
		},
	}

	backends := []backend.Backend{
		env.NewBackend(),
	}
	for _, cfgFile := range configs {
		backends = append(backends, file.NewBackend(cfgFile))
	}

	if err := confita.NewLoader(backends...).Load(context.Background(), cfg); err != nil {
		return nil, err
	}

	return cfg, cfg.Validate()
}
