package db

import (
	"fmt"
)

func lookupAuthorizationQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $id AS Utf8;

SELECT
	user, user_ticket, used, sign
FROM
	authorizations
WHERE
	id = $id
`, pathPrefix)
}

func requestMFAQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $authId AS Utf8;
DECLARE $user AS Utf8;
DECLARE $code AS Uint32;
DECLARE $requestedAt AS Uint64;

REPLACE INTO mfa
	(auth_id, user, requested_at, code)
VALUES
	($authId, $user, $requestedAt, $code)
;
`, pathPrefix)
}

func updateMFAQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $authId AS Utf8;
DECLARE $user AS Utf8;
DECLARE $tries AS Uint8;

UPDATE mfa
SET
	tries = $tries
WHERE
	auth_id = $authId AND user = $user
;
`, pathPrefix)
}

func lookupMFAQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $authId AS Utf8;
DECLARE $user AS Utf8;

SELECT
	requested_at, code, tries
FROM
	mfa
WHERE
	auth_id = $authId AND user = $user
LIMIT 1
`, pathPrefix)
}

func issueAuthorizationQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $id AS Utf8;
DECLARE $user AS Utf8;
DECLARE $userTicket AS Utf8;
DECLARE $issuedAt AS Int64;
DECLARE $sign AS Utf8;

UPSERT INTO authorizations
	(id, issued_at, user, user_ticket, sign, used)
VALUES
	($id, $issuedAt, $user, $userTicket, $sign, false);

DELETE FROM mfa WHERE auth_id = $id AND user = $user;
`, pathPrefix)
}

func dropAuthorizationQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $id AS Utf8;

UPDATE authorizations
SET
	used = True
WHERE
	id = $id
;
`, pathPrefix)
}

func lookupUserTokensQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $user AS Utf8;
DECLARE $revokedState AS Uint8;
DECLARE $limit AS Uint64;

SELECT
	id, enroll_id, name, token_type, token_state, created_at, updated_at, expires_at
FROM
	tokens
WHERE
	user = $user AND token_state != $revokedState
ORDER BY
	token_state ASC, created_at DESC
LIMIT $limit
`, pathPrefix)
}

func lookupActiveTokenSSHKeysQueryQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $tfid AS Utf8;
DECLARE $states AS List<Uint8>;
DECLARE $limit AS Uint64;

SELECT
	serial, cert_type, ssh_fingerprint
FROM
	certificates view certificates_tfid_index
WHERE
	tfid = $tfid AND cert_state in $states
LIMIT $limit
`, pathPrefix)
}

func lookupTokenCertsQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $tfid AS Utf8;
DECLARE $states AS List<Uint8>;
DECLARE $limit AS Uint64;

SELECT
	serial, cert_type, cert_state, created_at, valid_before, principal, ssh_fingerprint, ca_fingerprint
FROM
	certificates view certificates_tfid_index
WHERE
	tfid = $tfid AND cert_state in $states
ORDER BY
	cert_state ASC, serial DESC
LIMIT $limit
`, pathPrefix)
}

func lookupCertPubQuery(pathPrefix, column string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $serial AS Utf8;
DECLARE $tfid AS Utf8;
DECLARE $column AS Utf8;

SELECT
	%s
FROM
	certificates
WHERE
	serial = $serial AND tfid = $tfid
LIMIT 1
`, pathPrefix, column)
}

func issueCertificateQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $serial AS Utf8;
DECLARE $tfid AS Utf8;
DECLARE $certType AS Uint8;
DECLARE $certState AS Uint8;
DECLARE $createdAt AS Int64;
DECLARE $validAfter AS Int64;
DECLARE $validBefore AS Int64;
DECLARE $principal AS Utf8;
DECLARE $cert AS Utf8;
DECLARE $sshCert AS Utf8;
DECLARE $sshFingerprint AS Utf8;
DECLARE $caFingerprint AS Utf8;

INSERT INTO	certificates
	(serial, tfid, cert_type, cert_state, created_at, valid_after, valid_before, principal,  cert, ssh_cert, ssh_fingerprint, ca_fingerprint)
VALUES
	($serial, $tfid, $certType, $certState, $createdAt, $validAfter, $validBefore, $principal, $cert, $sshCert, $sshFingerprint, $caFingerprint)
;
`, pathPrefix)
}

func issueTokenQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $user AS Utf8;
DECLARE $id AS Utf8;
DECLARE $enrollId AS Utf8;
DECLARE $name AS Utf8;
DECLARE $tokenType AS Uint8;
DECLARE $tokenState AS Uint8;
DECLARE $createdAt AS Int64;
DECLARE $expiresAt AS Int64;

INSERT INTO	tokens
	(user, id, enroll_id, name, token_type, token_state, created_at, updated_at, expires_at)
VALUES
	($user, $id, $enrollId, $name, $tokenType, $tokenState, $createdAt, $createdAt, $expiresAt)
;
`, pathPrefix)
}

func updateTokenQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $user AS Utf8;
DECLARE $id AS Utf8;
DECLARE $enrollId AS Utf8;
DECLARE $name AS Utf8;
DECLARE $updatedAt AS Int64;
DECLARE $expiresAt AS Int64;

UPDATE tokens
SET
	name = $name, updated_at = $updatedAt, expires_at = $expiresAt
WHERE
	user = $user AND id = $id AND enroll_id = $enrollId
;
`, pathPrefix)
}

func lookupUserTokenState(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $user AS Utf8;
DECLARE $id AS Utf8;
DECLARE $enrollID AS Utf8;

SELECT
	token_state
FROM
	tokens
WHERE
	user = $user AND id = $id AND enroll_id = $enrollID
LIMIT 1
`, pathPrefix)
}

func scheduleRevokeTokenQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $user AS Utf8;
DECLARE $id AS Utf8;
DECLARE $enrollID AS Utf8;
DECLARE $updatedAt AS Int64;
DECLARE $fromState AS Uint8;
DECLARE $toState AS Uint8;

$tokens = (
	SELECT
    	user, id, enroll_id
	FROM tokens
	WHERE user = $user AND id = $id AND enroll_id = $enrollID AND token_state = $fromState
);

UPSERT INTO tokens SELECT user, id, enroll_id, $toState as token_state FROM $tokens;
UPSERT INTO tokens_to_revoke SELECT user, id, enroll_id, $updatedAt as revoked_at FROM $tokens;

SELECT user, id, enroll_id FROM $tokens;
`, pathPrefix)
}

func scheduleRevokeUserTokenQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $user AS Utf8;
DECLARE $id AS Utf8;
DECLARE $updatedAt AS Int64;
DECLARE $fromState AS Uint8;
DECLARE $toState AS Uint8;
DECLARE $minValidBefore AS Int64;

$tokens = (
	SELECT
    	user, id, enroll_id
	FROM tokens
	WHERE user = $user AND (id = $id  OR $id = "") AND token_state = $fromState AND expires_at > $minValidBefore
);

UPSERT INTO tokens SELECT user, id, enroll_id, $toState as token_state FROM $tokens;
UPSERT INTO tokens_to_revoke SELECT user, id, enroll_id, $updatedAt as revoked_at FROM $tokens;

SELECT user, id, enroll_id FROM $tokens;
`, pathPrefix)
}

func scheduleRevokeTokenIDQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $id AS Utf8;
DECLARE $updatedAt AS Int64;
DECLARE $fromState AS Uint8;
DECLARE $toState AS Uint8;
DECLARE $minValidBefore AS Int64;

$tokens = (
	SELECT
    	user, id, enroll_id
	FROM tokens view tokens_id_index
	WHERE id = $id AND token_state = $fromState AND expires_at > $minValidBefore
);

UPSERT INTO tokens SELECT user, id, enroll_id, $toState as token_state FROM $tokens;
UPSERT INTO tokens_to_revoke SELECT user, id, enroll_id, $updatedAt as revoked_at FROM $tokens;

SELECT user, id, enroll_id FROM $tokens;
`, pathPrefix)
}

func revokeTokenQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $user AS Utf8;
DECLARE $id AS Utf8;
DECLARE $enrollId AS Utf8;
DECLARE $tfid AS Utf8;
DECLARE $revokedAt AS Int64;
DECLARE $tokenState AS Uint8;
DECLARE $certState AS Uint8;
DECLARE $limit AS Uint64;

$certs = (
	SELECT
		serial, cert_type, ca_fingerprint, valid_before
	FROM
		certificates view certificates_tfid_index
	WHERE
		tfid = $tfid AND cert_state != $certState
	LIMIT $limit
);

UPSERT INTO certificates SELECT serial, $certState as cert_state FROM $certs;
UPSERT INTO revoked_certificates SELECT serial, ca_fingerprint, cert_type, valid_before, $revokedAt as revoked_at FROM $certs;

UPSERT INTO tokens
	(user, id, enroll_id, updated_at, token_state)
VALUES
	($user, $id, $enrollId, $revokedAt, $tokenState)
;

DELETE FROM tokens_to_revoke WHERE user = $user AND id = $id AND enroll_id = $enrollId;
`, pathPrefix)
}

func lookupRevokedTokensQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $maxRevokedAt AS Int64;
DECLARE $limit AS Uint64;

SELECT
	t.user as user, t.id as id, t.enroll_id as enroll_id, t.name as name, t.token_type as token_type
FROM tokens_to_revoke as r
INNER JOIN tokens as t USING(user, id, enroll_id)
WHERE revoked_at < $maxRevokedAt
LIMIT $limit;
`, pathPrefix)
}

func lookupExpiresTokensQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $minExpiresAt AS Int64;
DECLARE $maxExpiresAt AS Int64;
DECLARE $tokenState AS Uint8;
DECLARE $tokenType AS Uint8;
DECLARE $limit AS Uint64;

SELECT
	user, id, enroll_id, name, token_type, expires_at
FROM tokens
WHERE expires_at >= $minExpiresAt and expires_at <= $maxExpiresAt and token_state = $tokenState and token_type = $tokenType
LIMIT $limit;
`, pathPrefix)
}

func revokeCertsQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $tfid AS Utf8;
DECLARE $types AS List<Uint8>;
DECLARE $excludedSerials AS List<Utf8>;
DECLARE $fromState AS Uint8;
DECLARE $toState AS Uint8;
DECLARE $limit AS Uint64;
DECLARE $revokedAt AS Int64;

$certs = (
	SELECT
		serial, cert_type, ca_fingerprint, valid_before
	FROM
		certificates view certificates_tfid_index
	WHERE
		tfid = $tfid AND cert_state = $fromState AND cert_type IN $types AND serial NOT IN $excludedSerials
	LIMIT $limit
);

UPSERT INTO certificates SELECT serial, $toState as cert_state FROM $certs;
UPSERT INTO revoked_certificates SELECT serial, ca_fingerprint, cert_type, valid_before, $revokedAt as revoked_at FROM $certs;

SELECT serial, ca_fingerprint, cert_type FROM $certs;
`, pathPrefix)
}

func updateCertsStateQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $tfid AS Utf8;
DECLARE $types AS List<Uint8>;
DECLARE $excludedSerials AS List<Utf8>;
DECLARE $fromState AS Uint8;
DECLARE $toState AS Uint8;
DECLARE $limit AS Uint64;

$certs = (
	SELECT
		serial, cert_type, ca_fingerprint, valid_before
	FROM
		certificates view certificates_tfid_index
	WHERE
		tfid = $tfid AND cert_state = $fromState AND cert_type IN $types AND serial NOT IN $excludedSerials
	LIMIT $limit
);

UPSERT INTO certificates SELECT serial, $toState as cert_state FROM $certs;
SELECT serial, ca_fingerprint, cert_type FROM $certs;
`, pathPrefix)
}

func lookupAuditMsgQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $hash AS Uint64;
DECLARE $tokenId AS Utf8;
DECLARE $enrollId AS Utf8;
DECLARE $limit AS Uint64;

SELECT
	ts, message
FROM
	audit_log
WHERE
	hash = $hash AND token_id = $tokenId AND enroll_id = $enrollId
ORDER BY ts DESC
LIMIT $limit;
`, pathPrefix)
}

func insertAuditMsgQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $msgs AS List<Struct<
    hash: Uint64,
    token_id: Utf8,
    enroll_id: Utf8,
    ts: Int64,
    message: Utf8>>;

INSERT INTO	audit_log
SELECT hash, token_id, enroll_id, ts, message FROM AS_TABLE($msgs)
;
`, pathPrefix)
}

func lookupRevokedCertsQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");
DECLARE $lastSerial AS Utf8;
DECLARE $maxRevokedAt AS Int64;
DECLARE $limit AS Uint64;

SELECT
	serial, ca_fingerprint, cert_type
FROM revoked_certificates
WHERE
	serial > $lastSerial AND revoked_at < $maxRevokedAt AND valid_before > $maxRevokedAt
LIMIT $limit;
`, pathPrefix)
}

func createTablesQuery(pathPrefix string) string {
	return fmt.Sprintf(`
--!syntax_v1
PRAGMA TablePathPrefix("%s");

CREATE TABLE tokens
(
    user Utf8,
	id Utf8,
	enroll_id Utf8,
	name Utf8,
	token_type Uint8,
	token_state Uint8,
	created_at Int64,
	updated_at Int64,
	expires_at Int64,

	PRIMARY KEY (user, id, enroll_id),
	INDEX tokens_id_index GLOBAL ON (id)
);

CREATE TABLE tokens_to_revoke
(
	user Utf8,
	id Utf8,
	enroll_id Utf8,
	revoked_at Int64,

	PRIMARY KEY (user, id, enroll_id)
);

CREATE TABLE mfa
(
	auth_id Utf8,
	user Utf8,
	requested_at Uint64,
	code Uint32,
	tries Uint8,

	PRIMARY KEY (auth_id, user)
);
--WITH (
--    TTL = Interval("PT1H") ON requested_at
--);

CREATE TABLE authorizations
(
	id Utf8,
	issued_at Int64,
	user Utf8,
	user_ticket Utf8,
	sign Utf8,
	used Bool,

	PRIMARY KEY (id)
);

CREATE TABLE certificates
(
	serial Utf8,
	cert_type Uint8,
	cert_state Uint8,
	tfid Utf8,
	created_at Int64,
	valid_after Int64,
	valid_before Int64,
	principal Utf8,
	cert Utf8,
	ssh_cert Utf8,
	ssh_fingerprint Utf8,
	ca_fingerprint Utf8,

	PRIMARY KEY (serial),
	INDEX certificates_tfid_index GLOBAL ON (tfid)
);

CREATE TABLE revoked_certificates
(
	serial Utf8,
	ca_fingerprint Utf8,
	cert_type Uint8,
	valid_before Int64,
	revoked_at Int64,

	PRIMARY KEY (serial)
);

CREATE TABLE audit_log
(
	hash Uint64,
	token_id Utf8,
	enroll_id Utf8,
	ts Int64,
	message Utf8,

	PRIMARY KEY (hash, token_id, enroll_id, ts)
);
`, pathPrefix)
}
