package db

import (
	"context"
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/libs/go/ydbtvm"
	"a.yandex-team.ru/security/skotty/libs/skotty"
	"a.yandex-team.ru/security/skotty/service/internal/config"
	"a.yandex-team.ru/security/skotty/service/internal/models"
)

const (
	ListLimit = 999
)

type DB struct {
	sp   *table.SessionPool
	path string
}

var ErrNotFound = errors.New("not found")

func New(ctx context.Context, tvmc tvm.Client, cfg config.YDB) (*DB, error) {
	driverConfig := &ydb.DriverConfig{
		Database: cfg.Database,
		Credentials: &ydbtvm.TvmCredentials{
			DstID:     ydbtvm.YDBClientID,
			TvmClient: tvmc,
		},
	}

	driver, err := (&ydb.Dialer{
		DriverConfig: driverConfig,
	}).Dial(ctx, cfg.Endpoint)

	if err != nil {
		return nil, fmt.Errorf("dial error: %v", err)
	}

	tableClient := table.Client{
		Driver: driver,
	}

	sp := table.SessionPool{
		IdleThreshold: 10 * time.Second,
		Builder:       &tableClient,
	}

	err = createTables(ctx, &sp, cfg.Path)
	if err != nil {
		return nil, fmt.Errorf("create tables error: %v", err)
	}

	return &DB{
		sp:   &sp,
		path: cfg.Path,
	}, nil
}

func (d *DB) Close(ctx context.Context) error {
	return d.sp.Close(ctx)
}
func (d *DB) LookupAuthorization(ctx context.Context, id string) (*models.Authorization, error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	query := lookupAuthorizationQuery(d.path)
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$id", ydb.UTF8Value(id)),
			))
			return
		}),
	)

	if err != nil {
		return nil, err
	}

	if !res.NextSet() || !res.NextRow() {
		return nil, ErrNotFound
	}

	out := models.Authorization{
		ID: id,
	}

	// user, used, sign
	res.SeekItem("user")
	out.User = res.OUTF8()

	res.NextItem()
	out.UserTicket = res.OUTF8()

	res.NextItem()
	out.Used = res.OBool()

	res.NextItem()
	out.Sign = res.OUTF8()

	return &out, res.Err()
}

func (d *DB) RequestMFA(ctx context.Context, authID, user string, code uint32) error {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	query := requestMFAQuery(d.path)
	requestedAt := uint64(time.Now().Unix())
	return table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$authId", ydb.UTF8Value(authID)),
				table.ValueParam("$user", ydb.UTF8Value(user)),
				table.ValueParam("$code", ydb.Uint32Value(code)),
				table.ValueParam("$requestedAt", ydb.Uint64Value(requestedAt)),
			))
			return err
		}),
	)
}

func (d *DB) UpdateMFATries(ctx context.Context, authID, user string, tries uint8) error {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	query := updateMFAQuery(d.path)
	return table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$authId", ydb.UTF8Value(authID)),
				table.ValueParam("$user", ydb.UTF8Value(user)),
				table.ValueParam("$tries", ydb.Uint8Value(tries)),
			))
			return err
		}),
	)
}

func (d *DB) LookupMFA(ctx context.Context, authID, user string) (*models.MFA, error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	query := lookupMFAQuery(d.path)
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$authId", ydb.UTF8Value(authID)),
				table.ValueParam("$user", ydb.UTF8Value(user)),
			))
			return
		}),
	)

	if err != nil {
		return nil, err
	}

	if !res.NextSet() || !res.NextRow() {
		return nil, ErrNotFound
	}

	result := &models.MFA{
		AuthID: authID,
		User:   user,
	}
	// requested_at, code, tries
	res.SeekItem("requested_at")
	result.RequestedAt = res.OUint64()

	res.NextItem()
	result.Code = res.OUint32()

	res.NextItem()
	result.Tries = res.OUint8()

	return result, res.Err()
}

func (d *DB) IssueAuthorization(ctx context.Context, authInfo *models.Authorization) error {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	query := issueAuthorizationQuery(d.path)
	issuedAt := time.Now().Unix()
	return table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$id", ydb.UTF8Value(authInfo.ID)),
				table.ValueParam("$user", ydb.UTF8Value(authInfo.User)),
				table.ValueParam("$sign", ydb.UTF8Value(authInfo.Sign)),
				table.ValueParam("$userTicket", ydb.UTF8Value(authInfo.UserTicket)),
				table.ValueParam("$issuedAt", ydb.Int64Value(issuedAt)),
			))
			return err
		}),
	)
}

func (d *DB) DropAuthorization(ctx context.Context, authID string) error {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	query := dropAuthorizationQuery(d.path)
	return table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$id", ydb.UTF8Value(authID)),
			))
			return err
		}),
	)
}

func (d *DB) InsertCertificate(ctx context.Context, cert models.Certificate) error {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	query := issueCertificateQuery(d.path)
	createdAt := time.Now().Unix()
	return table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$serial", ydb.UTF8Value(cert.Serial)),
				table.ValueParam("$tfid", ydb.UTF8Value(cert.TokenFullID)),
				table.ValueParam("$certType", ydb.Uint8Value(uint8(cert.CertType))),
				table.ValueParam("$certState", ydb.Uint8Value(uint8(models.CertStateActive))),
				table.ValueParam("$createdAt", ydb.Int64Value(createdAt)),
				table.ValueParam("$validAfter", ydb.Int64Value(cert.ValidAfter)),
				table.ValueParam("$validBefore", ydb.Int64Value(cert.ValidBefore)),
				table.ValueParam("$principal", ydb.UTF8Value(cert.Principal)),
				table.ValueParam("$cert", ydb.UTF8Value(string(cert.Cert))),
				table.ValueParam("$sshCert", ydb.UTF8Value(string(cert.SSHCert))),
				table.ValueParam("$sshFingerprint", ydb.UTF8Value(cert.SSHFingerprint)),
				table.ValueParam("$caFingerprint", ydb.UTF8Value(cert.CAFingerprint)),
			))
			return err
		}),
	)
}

func (d *DB) InsertToken(ctx context.Context, token models.Token) error {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	query := issueTokenQuery(d.path)
	createdAt := time.Now().Unix()
	return table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$user", ydb.UTF8Value(token.User)),
				table.ValueParam("$id", ydb.UTF8Value(token.ID)),
				table.ValueParam("$enrollId", ydb.UTF8Value(token.EnrollID)),
				table.ValueParam("$name", ydb.UTF8Value(token.Name)),
				table.ValueParam("$tokenType", ydb.Uint8Value(uint8(token.TokenType))),
				table.ValueParam("$tokenState", ydb.Uint8Value(uint8(token.TokenState))),
				table.ValueParam("$createdAt", ydb.Int64Value(createdAt)),
				table.ValueParam("$expiresAt", ydb.Int64Value(token.ExpiresAt)),
			))
			return err
		}),
	)
}

func (d *DB) UpdateToken(ctx context.Context, token models.Token) error {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	query := updateTokenQuery(d.path)
	updatedAt := time.Now().Unix()
	return table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$user", ydb.UTF8Value(token.User)),
				table.ValueParam("$id", ydb.UTF8Value(token.ID)),
				table.ValueParam("$enrollId", ydb.UTF8Value(token.EnrollID)),
				table.ValueParam("$name", ydb.UTF8Value(token.Name)),
				table.ValueParam("$updatedAt", ydb.Int64Value(updatedAt)),
				table.ValueParam("$expiresAt", ydb.Int64Value(token.ExpiresAt)),
			))
			return err
		}),
	)
}

func (d *DB) SetCertsState(ctx context.Context, tokenFullID string, certs map[skotty.CertType]string, state models.CertState) ([]models.Certificate, error) {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	ydbCertTypes := make([]ydb.Value, len(certs))
	ydbCertSerials := make([]ydb.Value, len(certs))
	i := 0
	for typ, serial := range certs {
		ydbCertTypes[i] = ydb.Uint8Value(uint8(typ))
		ydbCertSerials[i] = ydb.UTF8Value(serial)
		i++
	}

	query := updateCertsStateQuery(d.path)
	var res *table.Result
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$tfid", ydb.UTF8Value(tokenFullID)),
				table.ValueParam("$types", ydb.ListValue(ydbCertTypes...)),
				table.ValueParam("$excludedSerials", ydb.ListValue(ydbCertSerials...)),
				table.ValueParam("$fromState", ydb.Uint8Value(uint8(models.CertStateActive))),
				table.ValueParam("$toState", ydb.Uint8Value(uint8(state))),
				table.ValueParam("$limit", ydb.Uint64Value(ListLimit)),
			))
			return
		}),
	)

	if err != nil {
		return nil, err
	}

	result := make([]models.Certificate, 0, ListLimit)
	for res.NextSet() {
		for res.NextRow() {
			var cert models.Certificate
			// serial, ca_fingerprint, cert_type
			res.SeekItem("serial")
			cert.Serial = res.OUTF8()

			res.NextItem()
			cert.CAFingerprint = res.OUTF8()

			res.NextItem()
			cert.CertType = models.CertType(res.OUint8())

			result = append(result, cert)
		}
	}

	return result, res.Err()
}

func (d *DB) RevokeCerts(ctx context.Context, tokenFullID string, certs map[skotty.CertType]string) ([]models.Certificate, error) {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	ydbCertTypes := make([]ydb.Value, len(certs))
	ydbCertSerials := make([]ydb.Value, len(certs))
	i := 0
	for typ, serial := range certs {
		ydbCertTypes[i] = ydb.Uint8Value(uint8(typ))
		ydbCertSerials[i] = ydb.UTF8Value(serial)
		i++
	}

	query := revokeCertsQuery(d.path)
	revokedAt := time.Now().Unix()
	var res *table.Result
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$tfid", ydb.UTF8Value(tokenFullID)),
				table.ValueParam("$types", ydb.ListValue(ydbCertTypes...)),
				table.ValueParam("$excludedSerials", ydb.ListValue(ydbCertSerials...)),
				table.ValueParam("$fromState", ydb.Uint8Value(uint8(models.CertStateActive))),
				table.ValueParam("$toState", ydb.Uint8Value(uint8(models.CertStateRevoked))),
				table.ValueParam("$revokedAt", ydb.Int64Value(revokedAt)),
				table.ValueParam("$limit", ydb.Uint64Value(ListLimit)),
			))
			return
		}),
	)

	if err != nil {
		return nil, err
	}

	result := make([]models.Certificate, 0, ListLimit)
	for res.NextSet() {
		for res.NextRow() {
			var cert models.Certificate
			// serial, ca_fingerprint, cert_type
			res.SeekItem("serial")
			cert.Serial = res.OUTF8()

			res.NextItem()
			cert.CAFingerprint = res.OUTF8()

			res.NextItem()
			cert.CertType = models.CertType(res.OUint8())

			result = append(result, cert)
		}
	}

	return result, res.Err()
}

func (d *DB) ScheduleRevokeToken(ctx context.Context, user, tokenID, enrollID string) ([]models.TokenID, error) {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	query := scheduleRevokeTokenQuery(d.path)
	now := time.Now().Unix()
	var res *table.Result
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$user", ydb.UTF8Value(user)),
				table.ValueParam("$id", ydb.UTF8Value(tokenID)),
				table.ValueParam("$enrollID", ydb.UTF8Value(enrollID)),
				table.ValueParam("$updatedAt", ydb.Int64Value(now)),
				table.ValueParam("$fromState", ydb.Uint8Value(uint8(models.TokenStateActive))),
				table.ValueParam("$toState", ydb.Uint8Value(uint8(models.TokenStateRevoking))),
			))
			return
		}),
	)

	if err != nil {
		return nil, err
	}

	result := make([]models.TokenID, 0, ListLimit)
	for res.NextSet() {
		for res.NextRow() {
			var token models.TokenID
			// user, id, enroll_id
			res.SeekItem("user")
			token.User = res.OUTF8()

			res.NextItem()
			token.ID = res.OUTF8()

			res.NextItem()
			token.EnrollID = res.OUTF8()

			result = append(result, token)
		}
	}

	return result, res.Err()
}

func (d *DB) ScheduleRevokeUserToken(ctx context.Context, user, tokenID string) ([]models.TokenID, error) {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	query := scheduleRevokeUserTokenQuery(d.path)
	now := time.Now().Unix()
	var res *table.Result
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$user", ydb.UTF8Value(user)),
				table.ValueParam("$id", ydb.UTF8Value(tokenID)),
				table.ValueParam("$updatedAt", ydb.Int64Value(now)),
				table.ValueParam("$fromState", ydb.Uint8Value(uint8(models.TokenStateActive))),
				table.ValueParam("$toState", ydb.Uint8Value(uint8(models.TokenStateRevoking))),
				table.ValueParam("$minValidBefore", ydb.Int64Value(now)),
			))
			return
		}),
	)

	if err != nil {
		return nil, err
	}

	result := make([]models.TokenID, 0, ListLimit)
	for res.NextSet() {
		for res.NextRow() {
			var token models.TokenID
			// user, id, enroll_id
			res.SeekItem("user")
			token.User = res.OUTF8()

			res.NextItem()
			token.ID = res.OUTF8()

			res.NextItem()
			token.EnrollID = res.OUTF8()

			result = append(result, token)
		}
	}

	return result, res.Err()
}

func (d *DB) ScheduleRevokeTokenID(ctx context.Context, tokenID string) ([]models.TokenID, error) {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	query := scheduleRevokeTokenIDQuery(d.path)
	now := time.Now().Unix()
	var res *table.Result
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$id", ydb.UTF8Value(tokenID)),
				table.ValueParam("$updatedAt", ydb.Int64Value(now)),
				table.ValueParam("$fromState", ydb.Uint8Value(uint8(models.TokenStateActive))),
				table.ValueParam("$toState", ydb.Uint8Value(uint8(models.TokenStateRevoking))),
				table.ValueParam("$minValidBefore", ydb.Int64Value(now)),
			))
			return
		}),
	)

	if err != nil {
		return nil, err
	}

	result := make([]models.TokenID, 0, ListLimit)
	for res.NextSet() {
		for res.NextRow() {
			var token models.TokenID
			// user, id, enroll_id
			res.SeekItem("user")
			token.User = res.OUTF8()

			res.NextItem()
			token.ID = res.OUTF8()

			res.NextItem()
			token.EnrollID = res.OUTF8()

			result = append(result, token)
		}
	}

	return result, res.Err()
}

func (d *DB) InsertAuditMsgs(ctx context.Context, msgs ...models.AuditMsg) error {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	ydbMsgs := make([]ydb.Value, len(msgs))
	for i, msg := range msgs {
		ydbMsgs[i] = ydb.StructValue(
			ydb.StructFieldValue("hash", ydb.Uint64Value(models.AuditMsgHash(msg.TokenID, msg.EnrollID))),
			ydb.StructFieldValue("token_id", ydb.UTF8Value(msg.TokenID)),
			ydb.StructFieldValue("enroll_id", ydb.UTF8Value(msg.EnrollID)),
			ydb.StructFieldValue("ts", ydb.Int64Value(msg.TS)),
			ydb.StructFieldValue("message", ydb.UTF8Value(msg.Message)),
		)
	}

	query := insertAuditMsgQuery(d.path)
	return table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$msgs", ydb.ListValue(ydbMsgs...)),
			))
			return err
		}),
	)
}

func (d *DB) LookupUserTokens(ctx context.Context, user string) ([]models.Token, error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	query := lookupUserTokensQuery(d.path)
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$user", ydb.UTF8Value(user)),
				table.ValueParam("$revokedState", ydb.Uint8Value(uint8(models.TokenStateRevoked))),
				table.ValueParam("$limit", ydb.Uint64Value(ListLimit)),
			))
			return
		}),
	)

	if err != nil {
		return nil, err
	}

	result := make([]models.Token, 0, ListLimit)
	for res.NextSet() {
		for res.NextRow() {
			token := models.Token{
				User: user,
			}

			// id, enroll_id, name, token_type, token_state, created_at, updated_at, expires_at
			res.SeekItem("id")
			token.ID = res.OUTF8()

			res.NextItem()
			token.EnrollID = res.OUTF8()

			res.NextItem()
			token.Name = res.OUTF8()

			res.NextItem()
			token.TokenType = models.TokenType(res.OUint8())

			res.NextItem()
			token.TokenState = models.TokenState(res.OUint8())

			res.NextItem()
			token.CreatedAt = res.OInt64()

			res.NextItem()
			token.UpdatedAt = res.OInt64()

			res.NextItem()
			token.ExpiresAt = res.OInt64()

			result = append(result, token)
		}
	}

	return result, res.Err()
}

func (d *DB) LookupUserTokenState(ctx context.Context, user, tokenID, enrollID string) (models.TokenState, error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	query := lookupUserTokenState(d.path)
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$user", ydb.UTF8Value(user)),
				table.ValueParam("$id", ydb.UTF8Value(tokenID)),
				table.ValueParam("$enrollID", ydb.UTF8Value(enrollID)),
			))
			return
		}),
	)
	if err != nil {
		return models.TokenStateNone, err
	}

	if !res.NextSet() || !res.NextRow() {
		return models.TokenStateNone, ErrNotFound
	}

	// token_state
	res.SeekItem("token_state")
	return models.TokenState(res.OUint8()), res.Err()
}

func (d *DB) LookupRevokedTokens(ctx context.Context, maxRevokedAt int64) ([]models.Token, error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	query := lookupRevokedTokensQuery(d.path)
	var res *table.Result
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$maxRevokedAt", ydb.Int64Value(maxRevokedAt)),
				table.ValueParam("$limit", ydb.Uint64Value(ListLimit)),
			))
			return err
		}),
	)

	if err != nil {
		return nil, err
	}

	result := make([]models.Token, 0, ListLimit)
	for res.NextSet() {
		for res.NextRow() {
			var token models.Token

			// t.user as user, t.id as id, t.enroll_id as enroll_id, t.name as name, t.token_type as token_type
			res.SeekItem("user")
			token.User = res.OUTF8()

			res.NextItem()
			token.ID = res.OUTF8()

			res.NextItem()
			token.EnrollID = res.OUTF8()

			res.NextItem()
			token.Name = res.OUTF8()

			res.NextItem()
			token.TokenType = models.TokenType(res.OUint8())

			result = append(result, token)
		}
	}

	return result, res.Err()
}

func (d *DB) LookupExpiresTokens(ctx context.Context, expiredAt time.Time) ([]models.Token, error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	query := lookupExpiresTokensQuery(d.path)
	var res *table.Result
	minExpiredAt := time.Date(expiredAt.Year(), expiredAt.Month(), expiredAt.Day(), 0, 0, 0, 0, expiredAt.Location()).Unix()
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$minExpiresAt", ydb.Int64Value(minExpiredAt)),
				table.ValueParam("$maxExpiresAt", ydb.Int64Value(minExpiredAt+86399)),
				table.ValueParam("$tokenState", ydb.Uint8Value(uint8(models.TokenStateActive))),
				// TODO(buglloc): need to remind about soft tokens?
				table.ValueParam("$tokenType", ydb.Uint8Value(uint8(models.TokenTypeYubikey))),
				table.ValueParam("$limit", ydb.Uint64Value(ListLimit)),
			))
			return err
		}),
	)

	if err != nil {
		return nil, err
	}

	result := make([]models.Token, 0, ListLimit)
	for res.NextSet() {
		for res.NextRow() {
			var token models.Token

			// user, id, enroll_id, name, token_type, expired_at
			res.SeekItem("user")
			token.User = res.OUTF8()

			res.NextItem()
			token.ID = res.OUTF8()

			res.NextItem()
			token.EnrollID = res.OUTF8()

			res.NextItem()
			token.Name = res.OUTF8()

			res.NextItem()
			token.TokenType = models.TokenType(res.OUint8())

			res.NextItem()
			token.ExpiresAt = res.OInt64()

			result = append(result, token)
		}
	}

	return result, res.Err()
}

func (d *DB) RevokeToken(ctx context.Context, token models.Token) error {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	query := revokeTokenQuery(d.path)
	revokedAt := time.Now().Unix()
	return table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$user", ydb.UTF8Value(token.User)),
				table.ValueParam("$id", ydb.UTF8Value(token.ID)),
				table.ValueParam("$enrollId", ydb.UTF8Value(token.EnrollID)),
				table.ValueParam("$tfid", ydb.UTF8Value(models.TFID(token.User, token.ID, token.EnrollID))),
				table.ValueParam("$revokedAt", ydb.Int64Value(revokedAt)),
				table.ValueParam("$tokenState", ydb.Uint8Value(uint8(models.TokenStateRevoked))),
				table.ValueParam("$certState", ydb.Uint8Value(uint8(models.CertStateRevoked))),
				table.ValueParam("$limit", ydb.Uint64Value(ListLimit)),
			))
			return err
		}),
	)
}

func (d *DB) LookupRevokedCerts(ctx context.Context, maxRevokedAt int64, lastCert models.Certificate) ([]models.Certificate, error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	query := lookupRevokedCertsQuery(d.path)
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$lastSerial", ydb.UTF8Value(lastCert.Serial)),
				table.ValueParam("$maxRevokedAt", ydb.Int64Value(maxRevokedAt)),
				table.ValueParam("$limit", ydb.Uint64Value(ListLimit)),
			))
			return
		}),
	)

	if err != nil {
		return nil, err
	}

	result := make([]models.Certificate, 0, ListLimit)
	for res.NextSet() {
		for res.NextRow() {
			var cert models.Certificate

			// serial, ca_fingerprint, cert_type
			res.SeekItem("serial")
			cert.Serial = res.OUTF8()

			res.NextItem()
			cert.CAFingerprint = res.OUTF8()

			res.NextItem()
			cert.CertType = models.CertType(res.OUint8())

			result = append(result, cert)
		}
	}

	return result, res.Err()
}

func (d *DB) LookupAuditMsgs(ctx context.Context, tokenID, enrollID string) ([]models.AuditMsg, error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	query := lookupAuditMsgQuery(d.path)
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$hash", ydb.Uint64Value(models.AuditMsgHash(tokenID, enrollID))),
				table.ValueParam("$tokenId", ydb.UTF8Value(tokenID)),
				table.ValueParam("$enrollId", ydb.UTF8Value(enrollID)),
				table.ValueParam("$limit", ydb.Uint64Value(ListLimit)),
			))
			return
		}),
	)

	if err != nil {
		return nil, err
	}

	result := make([]models.AuditMsg, 0, ListLimit)
	for res.NextSet() {
		for res.NextRow() {
			msg := models.AuditMsg{
				TokenID:  tokenID,
				EnrollID: enrollID,
			}

			// ts, message
			res.SeekItem("ts")
			msg.TS = res.OInt64()

			res.NextItem()
			msg.Message = res.OUTF8()

			result = append(result, msg)
		}
	}

	return result, res.Err()
}

func (d *DB) LookupTokenCertsQuery(ctx context.Context, tokenFullID string) ([]models.Certificate, error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	query := lookupTokenCertsQuery(d.path)
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$tfid", ydb.UTF8Value(tokenFullID)),
				table.ValueParam("$states", ydb.ListValue(
					ydb.Uint8Value(uint8(models.CertStateActive)),
					ydb.Uint8Value(uint8(models.CertStateRevoked)),
					ydb.Uint8Value(uint8(models.CertStateRenewed)),
				)),
				table.ValueParam("$limit", ydb.Uint64Value(ListLimit)),
			))
			return
		}),
	)

	if err != nil {
		return nil, err
	}

	result := make([]models.Certificate, 0, ListLimit)
	for res.NextSet() {
		for res.NextRow() {
			var cert models.Certificate

			// serial, cert_type, state, created_at, valid_before, principal, ssh_fingerprint, ca_fingerprint
			res.SeekItem("serial")
			cert.Serial = res.OUTF8()

			res.NextItem()
			cert.CertType = models.CertType(res.OUint8())

			res.NextItem()
			cert.CertState = models.CertState(res.OUint8())

			res.NextItem()
			cert.CreatedAt = res.OInt64()

			res.NextItem()
			cert.ValidBefore = res.OInt64()

			res.NextItem()
			cert.Principal = res.OUTF8()

			res.NextItem()
			cert.SSHFingerprint = res.OUTF8()

			res.NextItem()
			cert.CAFingerprint = res.OUTF8()

			result = append(result, cert)
		}
	}

	return result, res.Err()
}

func (d *DB) LookupActiveTokenSSHKeysQuery(ctx context.Context, tokenFullID string) ([]models.Certificate, error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	query := lookupActiveTokenSSHKeysQueryQuery(d.path)
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$tfid", ydb.UTF8Value(tokenFullID)),
				table.ValueParam("$states", ydb.ListValue(
					ydb.Uint8Value(uint8(models.CertStateActive)),
				)),
				table.ValueParam("$limit", ydb.Uint64Value(ListLimit)),
			))
			return
		}),
	)

	if err != nil {
		return nil, err
	}

	result := make([]models.Certificate, 0, ListLimit)
	for res.NextSet() {
		for res.NextRow() {
			var cert models.Certificate

			// serial, cert_type, ssh_cert, ssh_fingerprint
			res.SeekItem("serial")
			cert.Serial = res.OUTF8()

			res.NextItem()
			cert.CertType = models.CertType(res.OUint8())

			res.NextItem()
			cert.SSHFingerprint = res.OUTF8()

			result = append(result, cert)
		}
	}

	return result, res.Err()
}

func (d *DB) lookupCertPubQuery(ctx context.Context, serial, tfid, column string) (string, error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	query := lookupCertPubQuery(d.path, column)
	err := table.Retry(ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$serial", ydb.UTF8Value(serial)),
				table.ValueParam("$tfid", ydb.UTF8Value(tfid)),
			))
			return
		}),
	)

	if err != nil {
		return "", err
	}

	if !res.NextSet() || !res.NextRow() {
		return "", ErrNotFound
	}

	res.SeekItem(column)
	return res.OUTF8(), res.Err()
}

func (d *DB) LookupCertX509PubQuery(ctx context.Context, serial, tokenFullID string) (string, error) {
	return d.lookupCertPubQuery(ctx, serial, tokenFullID, "cert")
}

func (d *DB) LookupCertSSHPubQuery(ctx context.Context, serial, tokenFullID string) (string, error) {
	return d.lookupCertPubQuery(ctx, serial, tokenFullID, "ssh_cert")
}

func createTables(ctx context.Context, sp *table.SessionPool, prefix string) error {
	query := createTablesQuery(prefix)
	return table.Retry(ctx, sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			return s.ExecuteSchemeQuery(ctx, query)
		}),
	)
}
