package mailer

import (
	"bytes"
	"context"
	"fmt"
	"net/smtp"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/skotty/service/internal/models"
)

const (
	queueSize = 1024
)

type Mail struct {
	Subject   string
	Recipient string
	Template  string
	TmplData  map[string]interface{}
}

type Mailer struct {
	upstream  string
	from      string
	log       log.Logger
	ctx       context.Context
	cancelCtx context.CancelFunc
	buf       bytes.Buffer
	queue     chan Mail
	closed    chan struct{}
}

func NewMailer(upstream, from string, opts ...Option) *Mailer {
	ctx, cancelCtx := context.WithCancel(context.Background())
	m := &Mailer{
		upstream:  upstream,
		from:      from,
		ctx:       ctx,
		cancelCtx: cancelCtx,
		log:       &nop.Logger{},
		queue:     make(chan Mail, queueSize),
		closed:    make(chan struct{}),
	}

	for _, opt := range opts {
		opt(m)
	}

	go m.loop()
	return m
}

func (m *Mailer) TokenEnroll(ctx context.Context, token models.Token) {
	mail := Mail{
		Subject:   fmt.Sprintf("%q token has been successfully enrolled", token.Name),
		Template:  "token_enroll.gotmpl",
		Recipient: fmt.Sprintf("%s@yandex-team.ru", token.User),
		TmplData: map[string]interface{}{
			"ID":        token.ID,
			"Type":      token.TokenType.String(),
			"Name":      token.Name,
			"User":      token.User,
			"EnrollID":  token.EnrollID,
			"ExpiresAt": time.Unix(token.ExpiresAt, 0).Format(time.RFC822),
		},
	}

	select {
	case m.queue <- mail:
		return
	case <-ctx.Done():
		return
	}
}

func (m *Mailer) TokenRenew(ctx context.Context, token models.Token) {
	mail := Mail{
		Subject:   fmt.Sprintf("%q token certificates have been successfully renewed", token.Name),
		Template:  "token_renew.gotmpl",
		Recipient: fmt.Sprintf("%s@yandex-team.ru", token.User),
		TmplData: map[string]interface{}{
			"ID":        token.ID,
			"Type":      token.TokenType.String(),
			"Name":      token.Name,
			"User":      token.User,
			"EnrollID":  token.EnrollID,
			"ExpiresAt": time.Unix(token.ExpiresAt, 0).Format(time.RFC822),
		},
	}

	select {
	case m.queue <- mail:
		return
	case <-ctx.Done():
		return
	}
}

func (m *Mailer) TokenRevoked(ctx context.Context, token models.Token) {
	mail := Mail{
		Subject:   fmt.Sprintf("%q token has been revoked", token.Name),
		Template:  "token_revoked.gotmpl",
		Recipient: fmt.Sprintf("%s@yandex-team.ru", token.User),
		TmplData: map[string]interface{}{
			"ID":       token.ID,
			"Type":     token.TokenType.String(),
			"Name":     token.Name,
			"User":     token.User,
			"EnrollID": token.EnrollID,
		},
	}

	select {
	case m.queue <- mail:
		return
	case <-ctx.Done():
		return
	}
}

func (m *Mailer) TokenExpires(ctx context.Context, token models.Token, remainDays int) {
	expiresAt := time.Unix(token.ExpiresAt, 0).Format(time.RFC822)

	var subject string
	if remainDays == 0 {
		subject = fmt.Sprintf("%q token expires TODAY", token.Name)
	} else {
		subject = fmt.Sprintf("%q token is about to expire on %s", token.Name, expiresAt)
	}

	mail := Mail{
		Subject:   subject,
		Template:  "token_expires.gotmpl",
		Recipient: fmt.Sprintf("%s@yandex-team.ru", token.User),
		TmplData: map[string]interface{}{
			"ID":        token.ID,
			"Type":      token.TokenType.String(),
			"Name":      token.Name,
			"User":      token.User,
			"EnrollID":  token.EnrollID,
			"ExpiresAt": expiresAt,
		},
	}

	select {
	case m.queue <- mail:
		return
	case <-ctx.Done():
		return
	}
}

func (m *Mailer) Shutdown(ctx context.Context) {
	m.cancelCtx()

	select {
	case <-m.closed:
	case <-ctx.Done():
	}
}

func (m *Mailer) loop() {
	defer close(m.closed)

	for {
		select {
		case <-m.ctx.Done():
			return
		case mail := <-m.queue:
			if err := m.sendMail(mail); err != nil {
				m.log.Error("failed to send mail", log.String("recipient", mail.Recipient), log.Error(err))
			}
		}

	}
}

func (m *Mailer) sendMail(mail Mail) error {
	m.buf.Reset()

	writeHeader := func(key, val string) {
		m.buf.WriteString(key)
		m.buf.WriteString(": ")
		m.buf.WriteString(val)
		m.buf.WriteByte('\n')
	}

	writeHeader("From", fmt.Sprintf("Skotty<%s>", m.from))
	writeHeader("To", mail.Recipient)
	writeHeader("Subject", mail.Subject)
	writeHeader("Content-Type", `text/html; charset="utf-8"`)
	m.buf.WriteByte('\n')
	m.buf.WriteByte('\n')

	if err := renderTemplate(&m.buf, mail.Template, mail.TmplData); err != nil {
		return err
	}

	return smtp.SendMail(m.upstream, nil, m.from, []string{mail.Recipient}, m.buf.Bytes())
}
